<?php
// Load WordPress environment safely
require_once dirname(__FILE__) . '/index.php';

class WpPluginDirectoryShortDescriptionManagement
{

    public function __construct()
    {

        // Check & acquire lock
        if (! wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->updateShortDescriptions();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — updates active_installs in cron_scraped_plugins_log
     */
    private function updateShortDescriptions()
    {
        global $wpdb;

        $table_cron_page_visits = $wpdb->prefix . "cron_page_visits_log";
        $table_scraped_plugins  = $wpdb->prefix . "cron_scraped_plugins_log";

        // ✅ Fetch pages that need visiting
        $cron_page_visits = $wpdb->get_results("SELECT * FROM {$table_cron_page_visits} WHERE visited = 0");

        if (empty($cron_page_visits)) {
            echo "No unvisited pages found.\n";
            return;
        }

        foreach ($cron_page_visits as $single_object) {
            sleep(3); // throttle requests

            try {
                $response_data = $this->apiGet($single_object->endpoint);
            } catch (Exception $e) {
                error_log("API error for endpoint {$single_object->endpoint}: " . $e->getMessage());
                continue;
            }

            // ✅ Validate response
            if (! isset($response_data['plugins']) || empty($response_data['plugins'])) {
                error_log("No plugins found for endpoint: {$single_object->endpoint}");
                continue;
            }

            // ✅ Loop through plugins and update their active install counts
            foreach ($response_data['plugins'] as $plugin) {
                if (empty($plugin['slug'])) {
                    continue;
                }

                $update_data = [
                    'short_description' => $plugin['short_description'],
                ];

                $update_format = ['%s'];
                $where_data    = ['plugin_slug' => sanitize_text_field($plugin['slug'])];
                $where_format  = ['%s'];

                $result = $wpdb->update(
                    $table_scraped_plugins,
                    $update_data,
                    $where_data,
                    $update_format,
                    $where_format
                );

                if ($result === false) {
                    error_log("DB update failed for slug {$plugin['slug']}: " . $wpdb->last_error);
                }
            }

            // ✅ Mark this page as visited
            $wpdb->update(
                $table_cron_page_visits,
                [
                    'visited'         => 1,
                    'visit_count'     => $single_object->visit_count + 1,
                    'last_visited_at' => current_time('mysql'),
                ],
                [
                    'id' => $single_object->id,
                ],
                ['%d', '%d', '%s'],
                ['%d']
            );
        }
    }

    /**
     * Fetches JSON data via cURL
     */
    private function apiGet($url)
    {
        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT        => 30,
        ]);

        $response = curl_exec($curl);

        if (curl_errno($curl)) {
            throw new Exception('cURL error: ' . curl_error($curl));
        }

        $statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if ($statusCode !== 200) {
            throw new Exception("HTTP $statusCode returned for $url");
        }

        $decoded = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid JSON response from $url");
        }

        return $decoded;
    }
}

// Instantiate the class
new WpPluginDirectoryShortDescriptionManagement();
