<?php
// Load WordPress environment safely
require_once dirname(__FILE__) . '/index.php';

class WpPluginAgeCalculation
{

    public function __construct()
    {

        // Check & acquire lock
        if (! wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->pluginAgeCalculation();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — updates active_installs in cron_scraped_plugins_log
     */
    private function pluginAgeCalculation()
    {
        global $wpdb;

        update_option('wp_plugin_age_calculation_job_last_run', current_time('mysql'), false);
        $tbl_plugins_core_information = $wpdb->prefix . "plugins_core_information";

        // ✅ Fetch pages that need visiting
        $plugins_core_information = $wpdb->get_results(
            "SELECT id,last_updated,added
        FROM {$tbl_plugins_core_information}
        WHERE
        updated_at IS NULL
        OR DATEDIFF(NOW(), updated_at) > 7
        OR last_update_days = 0
        OR plugin_age_days = 0"
        );

        if (empty($plugins_core_information)) {
            echo "No Plugins found.\n";
            return;
        }

        foreach ($plugins_core_information as $single_plugin_information) {
            $last_update_since = $this->generateThePluginLastUpdateSinceDays($single_plugin_information->last_updated);
            $the_age_of_plugin = $this->generateThePluginAge($single_plugin_information->added);
            // ✅ Mark this page as visited
            $wpdb->update(
                $tbl_plugins_core_information,
                [
                    'last_update_days' => $last_update_since,
                    'plugin_age_days'  => $the_age_of_plugin,
                ],
                [
                    'id' => $single_plugin_information->id,
                ],
                ['%d', '%d'],
                ['%d']
            );
        }
        update_option('wp_plugin_age_calculation_job_last_run', current_time('mysql'), false);
    }
    private function generateThePluginLastUpdateSinceDays($last_update_since)
    {
        if (empty($last_update_since)) {
            return 0;
        }

        $updated_at = $last_update_since; // example from DB

        // Get current time (WordPress-safe)
        $current_time = current_time('mysql');

        // Convert both to timestamps
        $updated_timestamp = strtotime($updated_at);
        $current_timestamp = strtotime($current_time);

        // Calculate day difference
        return $days_since_update = floor(($current_timestamp - $updated_timestamp) / (60 * 60 * 24));
    }
    private function generateThePluginAge($date_string)
    {
        // Ensure valid date
        if (empty($date_string) || $date_string === '0000-00-00') {
            return null; // or 0 if you prefer
        }

        // Convert both to DateTime for accuracy
        $given_date = new DateTime($date_string);
        $today      = new DateTime();

        // Calculate the difference
        $interval = $today->diff($given_date);

        // Return the number of days difference
        return $interval->days;
    }
}

// Instantiate the class
new WpPluginAgeCalculation();
