<?php
class plugin_cpt_frontend_load
{
    public static function get_plugins_with_category_id_filter(
        $category_id,
        $limit = 20,
        $offset = 0,
        $dynamic_filters = [], // dynamic conditions
        $order_by = [],        // dynamic sorting
        $flag_column = null,
        $flag_value = null
    ) {
        global $wpdb;

        $tbl = $wpdb->prefix . "plugins_core_information";
        $terms_table = $wpdb->prefix . "term_relationships";

        /**
         * ---------------------------------------------------------
         * 0. Extract wp_post_id filter EARLY (optimized path)
         * ---------------------------------------------------------
         * Expecting wp_post_id list from fuzzy search / cache / index
         */
        $post_id_filter = null;

        foreach ($dynamic_filters as $key => $filter) {
            if (($filter['field'] ?? '') === 'wp_post_id' && !empty($filter['value'])) {
                $post_id_filter = (array) $filter['value'];

                // 🔐 Sanitize & normalize IDs (RECOMMENDED)
                $post_id_filter = array_values(array_unique(array_filter(
                    array_map('intval', $post_id_filter),
                    fn($id) => $id > 0
                )));

                unset($dynamic_filters[$key]); // remove from SQL filtering
                break;
            }
        }

        /**
         * ---------------------------------------------------------
         * 1. Get post_ids from category (term_relationships)
         * ---------------------------------------------------------
         */
        if (is_array($category_id) && !empty($category_id)) {
            $placeholders = implode(', ', array_fill(0, count($category_id), '%d'));
            $sql = $wpdb->prepare(
                "SELECT object_id FROM {$terms_table} WHERE term_taxonomy_id IN ($placeholders)",
                $category_id
            );
        } else {
            $sql = $wpdb->prepare(
                "SELECT object_id FROM {$terms_table} WHERE term_taxonomy_id = %d",
                $category_id
            );
        }

        $category_post_ids = array_map('intval', $wpdb->get_col($sql));

        if (empty($category_post_ids)) {
            return [];
        }

        /**
         * ---------------------------------------------------------
         * 2. INTERSECT category IDs with wp_post_id filter (FAST)
         * ---------------------------------------------------------
         */
        if (!empty($post_id_filter)) {
            $category_post_ids = array_values(array_intersect(
                $category_post_ids,
                $post_id_filter
            ));

            if (empty($category_post_ids)) {
                return [];
            }
        }

        $post_ids_sql = '(' . implode(',', $category_post_ids) . ')';

        /**
         * ---------------------------------------------------------
         * 3. Build base WHERE condition
         * ---------------------------------------------------------
         */
        $where_parts = [];
        $where_parts[] = "wp_post_id IN {$post_ids_sql}";

        /**
         * ---------------------------------------------------------
         * 4. Dynamic filtering system
         * ---------------------------------------------------------
         */
        $allowed_columns = [
            'rating',
            'downloads',
            'updated_at',
            'newest',
            'active_installs',
            'overall_wpscore',
            'performance_impact',
            'plugin_age_days',
            'verified',
            'is_sponsored',
            'is_featured',
            'status',
            'type',
            'support_quality',
            'user_review_sentiment',
            'localization_count',
            'last_update_days',
        ];

        foreach ($dynamic_filters as $filter) {

            if (empty($filter['field']) || empty($filter['operator'])) {
                continue;
            }

            $field = esc_sql($filter['field']);
            $operator = strtolower(trim($filter['operator']));
            $value = $filter['value'];

            if (!in_array($field, $allowed_columns, true)) {
                continue;
            }

            // BETWEEN
            if ($operator === 'between' && is_array($value) && count($value) === 2) {
                $where_parts[] = "{$field} BETWEEN " . floatval($value[0]) . " AND " . floatval($value[1]);
            }

            // > < >= <=
            elseif (in_array($operator, ['>', '<', '>=', '<='], true)) {
                $where_parts[] = "{$field} {$operator} " . floatval($value);
            }

            // = or !=
            elseif (in_array($operator, ['=', '!='], true)) {
                $where_parts[] = is_numeric($value)
                    ? "{$field} {$operator} " . intval($value)
                    : "{$field} {$operator} '" . esc_sql($value) . "'";
            }
        }

        /**
         * ---------------------------------------------------------
         * 5. Optional legacy flag filter
         * ---------------------------------------------------------
         */
        if (!empty($flag_column)) {
            $allowed_flags = ['is_sponsored', 'is_featured', 'status', 'type', 'verified'];

            if (in_array($flag_column, $allowed_flags, true)) {
                $where_parts[] = is_numeric($flag_value)
                    ? "{$flag_column} = " . intval($flag_value)
                    : "{$flag_column} = '" . esc_sql($flag_value) . "'";
            }
        }

        $where = "WHERE " . implode(" AND ", $where_parts);

        /**
         * ---------------------------------------------------------
         * 6. Dynamic ORDER BY system
         * ---------------------------------------------------------
         */
        $allowed_order_columns = [
            'rating',
            'active_installs',
            'overall_wpscore',
            'plugin_age_days',
            'performance_impact',
            'updated_at',
            'downloads',
        ];

        $order_parts = [];

        foreach ($order_by as $sort) {

            if (empty($sort['field'])) {
                continue;
            }

            if (!in_array($sort['field'], $allowed_order_columns, true)) {
                continue;
            }

            $direction = strtolower($sort['direction'] ?? 'desc');
            $direction = in_array($direction, ['asc', 'desc'], true) ? $direction : 'desc';

            $order_parts[] = esc_sql($sort['field']) . " {$direction}";
        }

        // Fallback → best plugins first
        if (empty($order_parts)) {
            $order_parts[] = "overall_wpscore DESC";
            $order_parts[] = "active_installs DESC";
        }

        $orderClause = "ORDER BY " . implode(', ', $order_parts);

        /**
         * ---------------------------------------------------------
         * 7. Select columns
         * ---------------------------------------------------------
         */
        $columns = "
        plugin_icon,
        post_title,
        author_name,
        author_profile,
        overall_wpscore,
        rating,
        num_ratings,
        active_installs,
        tested,
        wp_post_id
    ";

        /**
         * ---------------------------------------------------------
         * 8. Build final SQL
         * ---------------------------------------------------------
         */
        $sql = "SELECT {$columns} FROM {$tbl} {$where} {$orderClause}";

        if ($limit > 0) {
            $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", intval($limit), intval($offset));
        }
        //return $sql;
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }




    public static function get_plugins_with_category_id_filter_exclude(
        $category_id,
        $exclude_id,
        $limit,
        $offset,
        $filters = null,
        $flag_column = null,
        $flag_value = null
    ) {
        global $wpdb;
        $tbl = $wpdb->prefix . "plugins_core_information";
        $terms_table = $wpdb->prefix . "term_relationships";
        $post_ids = $wpdb->get_col($wpdb->prepare(
            "SELECT object_id FROM {$terms_table} WHERE term_taxonomy_id = %d",
            $category_id
        ));

        if (empty($post_ids)) {
            // No posts found for this category → return empty
            return [];
        }

        // Convert to SQL-safe IN() list: (1,2,3,4)
        $post_ids_sql = "(" . implode(",", array_map('intval', $post_ids)) . ")";
        // --- Sanitize inputs ---
        $category_id = esc_sql($category_id);
        $exclude_id = intval($exclude_id);
        $limit = intval($limit);
        $offset = intval($offset);

        // --- Build WHERE parts ---
        $where_parts = [];
        $where_parts[] = "wp_post_id IN {$post_ids_sql}";

        // Exclude specific ID (from wp_post_id)
        $where_parts[] = "id != {$exclude_id}";

        // ✅ Flag column filtering (supports numeric/string)
        if (!empty($flag_column)) {
            $allowed_flags = ['is_sponsored', 'is_featured', 'status', 'type', 'verified'];
            if (in_array($flag_column, $allowed_flags)) {
                $flag_column = esc_sql($flag_column);
                if (is_numeric($flag_value)) {
                    $where_parts[] = "{$flag_column} = " . intval($flag_value);
                } elseif (!is_null($flag_value)) {
                    $where_parts[] = "{$flag_column} = '" . esc_sql($flag_value) . "'";
                }
            }
        }

        // Combine WHERE
        $where = "WHERE " . implode(' AND ', $where_parts);

        // --- Build ORDER BY dynamically ---
        $allowed_filters = [
            'rating',
            'downloads',
            'updated_at',
            'newest',
            'active_installs',
            'overall_wpscore',
            'performance_impact',
            'plugin_age_days',
        ];
        $order_parts = [];

        if (!empty($filters)) {
            if (is_string($filters)) {
                $filters = [$filters];
            }

            foreach ($filters as $filter) {
                $parts = explode(':', $filter);
                $column = esc_sql(trim($parts[0]));
                $direction = isset($parts[1]) ? strtolower(trim($parts[1])) : 'desc';

                if (in_array($column, $allowed_filters)) {
                    $direction = in_array($direction, ['asc', 'desc']) ? $direction : 'desc';
                    $order_parts[] = "{$column} {$direction}";
                }
            }
        }

        // ✅ Always add active_installs DESC as fallback
        $order_parts[] = "active_installs DESC";

        $orderClause = "ORDER BY " . implode(', ', $order_parts);

        // --- Select columns ---
        $columns = "
        plugin_icon,
        post_title,
        author_name,
        author_profile,
        overall_wpscore,
        rating,
        num_ratings,
        active_installs,
        tested,
        wp_post_id
    ";

        // --- Build final SQL ---
        $sql = "SELECT {$columns} FROM {$tbl} {$where} {$orderClause}";

        // --- Add LIMIT/OFFSET ---
        if ($limit > 0) {
            $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $limit, $offset);
        }

        // --- Execute ---
        $results = $wpdb->get_results($sql, ARRAY_A);

        return $results ?: [];
    }
    public static function get_the_relevent_comments($slug, $type)
    {
        global $wpdb;
        $tbl_plugin_comments = $wpdb->prefix . "plugin_comments";

        // Prepare a query based on the type
        if ($type == "relevent") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY RAND() LIMIT 3",
                $slug
            );
        }
        if ($type == "newest") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY id ASC LIMIT 3",
                $slug
            );
        }
        if ($type == "highest") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY commenter_num_rating DESC, id ASC LIMIT 3",
                $slug
            );
        }
        if ($type == "lowest") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY commenter_num_rating ASC, id ASC LIMIT 3",
                $slug
            );
        }
        //return $sql;
        // Execute the query
        $results = $wpdb->get_results($sql);

        return $results;
    }
    public static function get_the_plugin_comments($slug)
    {
        global $wpdb;
        $tbl = $wpdb->prefix . "plugin_comments";

        // 1) Fetch latest 30 comments sorted by date
        $sql = $wpdb->prepare(
            "SELECT * FROM {$tbl}
         WHERE plugin_slug = %s
         ORDER BY STR_TO_DATE(comment_date, '%%M %%d, %%Y') DESC
         LIMIT 30",
            $slug
        );

        $rows = $wpdb->get_results($sql, ARRAY_A);

        if (empty($rows)) {
            return [
                "relevent" => [],
                "newest" => [],
                "highest" => [],
                "lowest" => [],
            ];
        }

        // Normalize fields
        $comments = [];
        foreach ($rows as $r) {
            $r['commenter_num_rating'] = (int) preg_replace('/\D/', '', $r['commenter_num_rating']);
            $r['id'] = (int) $r['id'];

            // Convert comment_date to timestamp using DateTime for reliability
            try {
                $date = DateTime::createFromFormat('F j, Y', $r['comment_date']);
                $r['comment_timestamp'] = $date ? $date->getTimestamp() : 0;
            } catch (Exception $e) {
                $r['comment_timestamp'] = 0;
            }

            $comments[] = $r;
        }

        // -----------------------------------------------
        // 2) Sort and limit to 3 most recent for each category
        // -----------------------------------------------

        // RELEVANT → 3 random from most recent comments
        $relevent = $comments;
        shuffle($relevent);
        $relevent = array_slice($relevent, 0, 3);

        // NEWEST → top 3 by comment_date (already sorted DESC from query)
        $newest = array_slice($comments, 0, 3);

        // HIGHEST → sort by rating DESC, then by most recent date for tie-breaker
        $highest = $comments;
        usort($highest, function ($a, $b) {
            if ($b['commenter_num_rating'] == $a['commenter_num_rating']) {
                return $b['comment_timestamp'] <=> $a['comment_timestamp']; // most recent first
            }
            return $b['commenter_num_rating'] <=> $a['commenter_num_rating'];
        });
        $highest = array_slice($highest, 0, 3);

        // LOWEST → sort by rating ASC, then by most recent date for tie-breaker
        $lowest = $comments;
        usort($lowest, function ($a, $b) {
            if ($a['commenter_num_rating'] == $b['commenter_num_rating']) {
                return $b['comment_timestamp'] <=> $a['comment_timestamp']; // most recent first
            }
            return $a['commenter_num_rating'] <=> $b['commenter_num_rating'];
        });
        $lowest = array_slice($lowest, 0, 3);

        // -----------------------------------------------
        // 3) RETURN ALL SORTED ARRAYS (MAX 3 EACH)
        // -----------------------------------------------
        return [
            "relevent" => array_values($relevent),
            "newest" => array_values($newest),
            "highest" => array_values($highest),
            "lowest" => array_values($lowest),
        ];
    }
    public static function get_plugins_by_ids(array $ids = [])
    {
        //return  $ids;
        global $wpdb;

        // Skip if empty
        if (empty($ids)) {
            return [];
        }

        // Force ints & clean
        $ids = array_map('intval', $ids);
        $ids = array_filter($ids);

        if (empty($ids)) {
            return [];
        }

        $table = $wpdb->prefix . "plugins_core_information";

        // Same columns you use in your main query
        $columns = "
        plugin_icon,
        post_title,
        author_name,
        author_profile,
        overall_wpscore,
        rating,
        num_ratings,
        active_installs,
        tested,
        post_short_descripton,
        wp_post_id
    ";

        // Build placeholders
        $placeholders = implode(', ', array_fill(0, count($ids), '%d'));

        // SQL
        $sql = $wpdb->prepare(
            "SELECT {$columns}
         FROM {$table}
         WHERE id IN ($placeholders)",
            $ids
        );
        //return $sql;
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }
    public static function fuzzy_search_plugin_suggestions_with_category($search_text)
    {
        if (empty($search_text)) {
            return [];
        }

        /* -------------------------
         * Load JSON suggestions
         * ------------------------- */
        $file_path = get_stylesheet_directory() . '/plugins-suggestions.json';
        if (!file_exists($file_path)) {
            return [];
        }

        $suggestions = json_decode(file_get_contents($file_path), true);
        if (empty($suggestions) || !is_array($suggestions)) {
            return [];
        }

        /* -------------------------
         * Load top categories option
         * ------------------------- */
        $top_categories = get_option('top_plugin_categories');
        if (empty($top_categories) || !is_array($top_categories)) {
            $top_categories = [];
        }

        /* -------------------------
         * Normalize helper
         * ------------------------- */
        $normalize = function ($string) {
            $string = html_entity_decode($string, ENT_QUOTES, 'UTF-8');
            $string = mb_strtolower($string);
            $string = preg_replace('/[^\p{L}\p{N}\s]/u', ' ', $string);
            return trim(preg_replace('/\s+/', ' ', $string));
        };

        $needle = $normalize($search_text);
        $needle_words = explode(' ', $needle);

        /* -------------------------
         * Prepare category index
         * ------------------------- */
        $category_index = [];

        foreach ($top_categories as $cat) {
            $category_index[] = [
                'id' => (int) $cat['term_id'],
                'name' => $normalize($cat['name'] ?? ''),
                'slug' => $normalize($cat['slug'] ?? ''),
            ];
        }

        $results = [];

        /* -------------------------
         * Fuzzy match suggestions
         * ------------------------- */
        foreach ($suggestions as $item) {

            // ❌ Skip category-only rows immediately
            if (empty($item['post_id'])) {
                continue;
            }

            $title_raw = $item['title'] ?? '';
            $text_raw = $item['text'] ?? '';

            $title = $normalize($title_raw);
            $text = $normalize($text_raw);

            $score = 0;

            // Strong matches
            if (strpos($title, $needle) !== false) {
                $score += 100;
            } elseif (strpos($text, $needle) !== false) {
                $score += 60;
            }

            // Word matches
            foreach ($needle_words as $word) {
                if (strlen($word) < 3) {
                    continue;
                }

                if (strpos($title, $word) !== false) {
                    $score += 25;
                } elseif (strpos($text, $word) !== false) {
                    $score += 10;
                }
            }

            if ($score < 30) {
                continue;
            }

            /* -------------------------
             * Infer category from content
             * ------------------------- */
            $best_category_id = null;
            $best_cat_score = 0;

            foreach ($category_index as $cat) {
                $cat_score = 0;

                if (strpos($title, $cat['name']) !== false) {
                    $cat_score += 50;
                }

                if (strpos($text, $cat['name']) !== false) {
                    $cat_score += 30;
                }

                if (strpos($title, $cat['slug']) !== false) {
                    $cat_score += 20;
                }

                if ($cat_score > $best_cat_score) {
                    $best_cat_score = $cat_score;
                    $best_category_id = $cat['id'];
                }
            }

            $results[] = [
                'post_id' => (int) $item['post_id'],
                'category_id' => $best_category_id,
                'score' => $score + $best_cat_score,
            ];
        }

        /* -------------------------
         * Sort by final score
         * ------------------------- */
        usort($results, function ($a, $b) {
            return $b['score'] <=> $a['score'];
        });

        /* -------------------------
         * Clean return
         * ------------------------- */
        return array_map(function ($item) {
            return [
                'post_id' => $item['post_id'],
                'category_id' => $item['category_id'],
            ];
        }, $results);

    }
    public static function fuzzy_search_plugin_suggestions_with_category_only_post($search_text, $filter_category_id)
    {
        if (empty($search_text) || empty($filter_category_id)) {
            return [];
        }

        /* -------------------------
         * Load JSON suggestions
         * ------------------------- */
        $file_path = get_stylesheet_directory() . '/plugins-suggestions.json';
        if (!file_exists($file_path)) {
            return [];
        }

        $suggestions = json_decode(file_get_contents($file_path), true);
        if (empty($suggestions) || !is_array($suggestions)) {
            return [];
        }

        /* -------------------------
         * Load top categories
         * ------------------------- */
        $top_categories = get_option('top_plugin_categories');
        if (empty($top_categories) || !is_array($top_categories)) {
            return [];
        }

        /* -------------------------
         * Normalize helper
         * ------------------------- */
        $normalize = function ($string) {
            $string = html_entity_decode($string, ENT_QUOTES, 'UTF-8');
            $string = mb_strtolower($string);
            $string = preg_replace('/[^\p{L}\p{N}\s]/u', ' ', $string);
            return trim(preg_replace('/\s+/', ' ', $string));
        };

        $needle = $normalize($search_text);
        $needle_words = explode(' ', $needle);

        /* -------------------------
         * Prepare category index
         * ------------------------- */
        $category_index = [];

        foreach ($top_categories as $cat) {
            $category_index[] = [
                'id' => (int) $cat['term_id'],
                'name' => $normalize($cat['name'] ?? ''),
                'slug' => $normalize($cat['slug'] ?? ''),
            ];
        }

        $results = [];

        /* -------------------------
         * Fuzzy match + category filter
         * ------------------------- */
        foreach ($suggestions as $item) {

            if (empty($item['post_id'])) {
                continue;
            }

            $title = $normalize($item['title'] ?? '');
            $text = $normalize($item['text'] ?? '');

            $score = 0;

            // Strong matches
            if (strpos($title, $needle) !== false) {
                $score += 100;
            } elseif (strpos($text, $needle) !== false) {
                $score += 60;
            }

            // Word matches
            foreach ($needle_words as $word) {
                if (strlen($word) < 3) {
                    continue;
                }

                if (strpos($title, $word) !== false) {
                    $score += 25;
                } elseif (strpos($text, $word) !== false) {
                    $score += 10;
                }
            }

            if ($score < 30) {
                continue;
            }

            /* -------------------------
             * Infer category
             * ------------------------- */
            $best_category_id = null;
            $best_cat_score = 0;

            foreach ($category_index as $cat) {

                $cat_score = 0;

                if (strpos($title, $cat['name']) !== false) {
                    $cat_score += 50;
                }

                if (strpos($text, $cat['name']) !== false) {
                    $cat_score += 30;
                }

                if (strpos($title, $cat['slug']) !== false) {
                    $cat_score += 20;
                }

                if ($cat_score > $best_cat_score) {
                    $best_cat_score = $cat_score;
                    $best_category_id = $cat['id'];
                }
            }

            // ✅ CATEGORY FILTER HERE
            if ((int) $best_category_id !== (int) $filter_category_id) {
                continue;
            }

            $results[] = [
                'post_id' => (int) $item['post_id'],
                'score' => $score + $best_cat_score,
            ];
        }

        /* -------------------------
         * Sort by relevance
         * ------------------------- */
        usort($results, function ($a, $b) {
            return $b['score'] <=> $a['score'];
        });

        /* -------------------------
         * Return only post IDs
         * ------------------------- */
        return array_values(array_unique(array_column($results, 'post_id')));
    }

}
