<?php
if (!defined('ABSPATH'))
    exit;

// Include WP_List_Table if not loaded
if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class Plugin_Scrap_Log_List_Table extends WP_List_Table
{

    /** Fetch data from DB with optional search and filters */
    public function fetch_data($search = '', $filters = [])
    {
        global $wpdb;
        $table = $wpdb->prefix . 'plugins_core_information';

        $sql = "SELECT 
                id,
                plugin_feature_image,
                plugin_icon,
                name,
                slug,
                version,
                active_installs,
                downloaded,
                author_name,
                author_profile,
                rating,
                updated_at,
                created_at,
                overall_wpscore,
                plugin_open_ai_category
            FROM $table";

        $where_clauses = [];

        // Search filter - search in name and slug
        if (!empty($search)) {
            $search = esc_sql($wpdb->esc_like($search));
            $where_clauses[] = "(name LIKE '%$search%' OR slug LIKE '%$search%')";
        }

        // Add category filter if present
        if (isset($filters['filter_plugin_category']) && !empty($filters['filter_plugin_category'])) {
            $category = esc_sql($wpdb->esc_like($filters['filter_plugin_category']));
            // Search in both category and subcategories fields
            $where_clauses[] = "(
                JSON_EXTRACT(plugin_open_ai_category, '$.category') LIKE '%$category%'
                OR JSON_EXTRACT(plugin_open_ai_category, '$.subcategories') LIKE '%$category%'
            )";
        }

        // Status filters - handle 1/0 values
        $filter_columns = [
            'is_api_crawled',
            'is_web_crawled',
            'is_meta_information_from_ai_search',
            'is_ai_long_description_done',
            'is_ai_meta_keywords_done',
            'is_ai_tax_category_done',
            'is_wp_post_created'
        ];

        foreach ($filter_columns as $col) {
            if (isset($filters[$col]) && $filters[$col] !== '') {
                $value = ($filters[$col] === 'done') ? '1' : '0';
                $where_clauses[] = "$col = " . intval($value);
            }
        }
        if (isset($filters['license_type']) && $filters['license_type'] !== '') {
            $license = esc_sql($filters['license_type']);
            $where_clauses[] = "license_type = '$license'";
        }

        if (!empty($where_clauses)) {
            $sql .= " WHERE " . implode(' AND ', $where_clauses);
        }

        // Default order by active_installs DESC
        $sql .= " ORDER BY active_installs DESC LIMIT 1000";

        $results = $wpdb->get_results($sql, ARRAY_A);
        return $results ?: [];
    }


    /** Define table columns */
    public function get_columns()
    {
        return [
            'plugin_feature_image' => 'Feature Image',
            'name' => 'Plugin Name',
            'version' => 'Version',
            'author_name' => 'Author',
            'active_installs' => 'Active Installed',
            'downloaded' => 'Downloaded',
            'rating' => 'Rating',
            'overall_wpscore' => 'Overall WPScore',
            'updated_at' => 'Updated At',
            'created_at' => 'Created At',
            'actions' => 'Actions',
        ];
    }

    /** Sortable columns */
    protected function get_sortable_columns()
    {
        return [
            'name' => ['name', true],
            'overall_wpscore' => ['overall_wpscore', true],
            'active_installs' => ['active_installs', true],
            'downloaded' => ['downloaded', true],
            'rating' => ['rating', false],
            'created_at' => ['created_at', false],
            'updated_at' => ['updated_at', false],
        ];
    }

    /** Feature Image column */
    protected function column_plugin_feature_image($item)
    {
        // Determine image source
        $image = !empty($item['plugin_icon'])
            ? $item['plugin_icon']
            : $item['plugin_feature_image'];

        if (!empty($image)) {
            return sprintf(
                '<img src="%s" alt="%s" style="width:100px;height:80px;object-fit:contain;border-radius:6px;">',
                esc_url($image),
                esc_attr($item['name'])
            );
        }

        return '<span style="color:#aaa;">No image</span>';
    }

    /** Autor column */
    protected function column_author_name($item)
    {
        // Determine image source
        return sprintf(
            '<a href="%s" target="_blank">%s</a>',
            esc_url($item['author_profile']),
            esc_html($item['author_name'])
        );
    }

    /** Actions column */
    protected function column_actions($item)
    {
        $view_url = add_query_arg([
            'page' => 'plugin-scrap-log-details',
            'slug' => $item['slug']
        ], admin_url('admin.php'));

        return sprintf(
            '<a href="%s" class="button button-small button-primary">View Details</a>',
            esc_url($view_url)
        );
    }

    /** Default text columns */
    protected function column_default($item, $column_name)
    {
        return esc_html($item[$column_name] ?? '');
    }

    /** Prepare items */
    public function prepare_items()
    {
        $columns = $this->get_columns();
        $hidden = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [$columns, $hidden, $sortable];

        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';

        // Get filter values - FIXED: Added filter_plugin_category
        $filters = [
            'is_api_crawled' => $_REQUEST['filter_is_api_crawled'] ?? '',
            'is_web_crawled' => $_REQUEST['filter_is_web_crawled'] ?? '',
            'is_meta_information_from_ai_search' => $_REQUEST['filter_is_meta_information_from_ai_search'] ?? '',
            // 'is_ai_long_description_done' => $_REQUEST['filter_is_ai_long_description_done'] ?? '',
            // 'is_ai_meta_keywords_done' => $_REQUEST['filter_is_ai_meta_keywords_done'] ?? '',
            'is_ai_tax_category_done' => $_REQUEST['filter_is_ai_tax_category_done'] ?? '',
            'is_wp_post_created' => $_REQUEST['filter_is_wp_post_created'] ?? '',
            'filter_plugin_category' => $_REQUEST['filter_plugin_category'] ?? '',
            'license_type' => $_REQUEST['filter_license_type'] ?? '',
        ];

        $filters = array_map('sanitize_text_field', $filters);
        $data = $this->fetch_data($search, $filters);

        // Default orderby = active_installs, order = desc
        $orderby = $_REQUEST['orderby'] ?? 'active_installs';
        $order = $_REQUEST['order'] ?? 'desc';

        usort($data, function ($a, $b) use ($orderby, $order) {
            $a_val = $a[$orderby] ?? '';
            $b_val = $b[$orderby] ?? '';
            $result = strnatcmp($a_val, $b_val);
            return ($order === 'asc') ? $result : -$result;
        });

        // Pagination items-per-page dropdown
        $per_page = isset($_REQUEST['per_page']) ? intval($_REQUEST['per_page']) : 20;
        if (!in_array($per_page, [20, 50, 100, 200])) {
            $per_page = 20;
        }

        $current_page = $this->get_pagenum();
        $total_items = count($data);
        $this->items = array_slice($data, (($current_page - 1) * $per_page), $per_page);

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page' => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ]);
    }

    /** Extra tablenav - Add filters above the table */
    protected function extra_tablenav($which)
    {
        if ($which === 'top') {
            // Enqueue Select2
            wp_enqueue_style('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css');
            wp_enqueue_script('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', ['jquery'], null, true);

            $filter_names = [
                'filter_is_api_crawled',
                'filter_is_web_crawled',
                'filter_is_meta_information_from_ai_search',
                // 'filter_is_ai_long_description_done',
                // 'filter_is_ai_meta_keywords_done',
                'filter_is_ai_tax_category_done',
                'filter_is_wp_post_created',
                'filter_plugin_category',
                'filter_license_type' => 'License Type',
            ];
            ?>
            <div id="filter-panel"
                style="margin-top:10px; margin-bottom:15px; clear:both; background:#f9f9f9; padding:15px; border:1px solid #ddd; border-radius:4px;">
                <h3 style="margin-top:0;">
                    <span class="dashicons dashicons-filter" style="margin-top:3px;"></span> Filters
                </h3>
                <div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap:15px;">
                    <div>
                        <label style="display:block; margin-bottom:5px; font-weight:600;">Search</label>
                        <input type="search" name="s" value="<?php echo esc_attr($_REQUEST['s'] ?? ''); ?>"
                            placeholder="Search by name or slug..." style="width:100%;" />
                    </div>
                    <?php
                    // Existing filter options
                    $filters = [
                        'filter_is_api_crawled' => 'API Crawled',
                        'filter_is_web_crawled' => 'Web Crawled',
                        'filter_is_meta_information_from_ai_search' => 'License,Pro Meta Info Done',
                        // 'filter_is_ai_long_description_done' => 'AI Long Desc',
                        // 'filter_is_ai_meta_keywords_done' => 'AI Keywords',
                        'filter_is_ai_tax_category_done' => 'AI Taxonomy',
                        'filter_is_wp_post_created' => 'WP Post Created',
                    ];
                    $categories = explode(',', get_option('pre_defined_ai_category'));
                    ?>
                    <div>
                        <label style="display:block; margin-bottom:5px; font-weight:600;">Category</label>
                        <select name="filter_plugin_category" id="filter_plugin_category" style="width:100%;">
                            <option value="">All Categories</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo esc_attr($category); ?>" <?php selected($_REQUEST['filter_plugin_category'] ?? '', $category); ?>>
                                    <?php echo esc_html($category); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php
                    foreach ($filters as $name => $label) {
                        $current_value = isset($_REQUEST[$name]) ? sanitize_text_field($_REQUEST[$name]) : '';
                        ?>
                        <div>
                            <label style="display:block; margin-bottom:5px; font-weight:600;"><?php echo esc_html($label); ?></label>
                            <select name="<?php echo esc_attr($name); ?>" style="width:100%;">
                                <option value="">All</option>
                                <option value="done" <?php selected($current_value, 'done'); ?>>Done</option>
                                <option value="not_done" <?php selected($current_value, 'not_done'); ?>>Not Done</option>
                            </select>
                        </div>
                        <?php
                    }
                    ?>
                    <div>
                        <label style="display:block; margin-bottom:5px; font-weight:600;">License Type</label>
                        <select name="filter_license_type" style="width:100%;">
                            <option value="">All</option>
                            <option value="free" <?php selected($_REQUEST['filter_license_type'] ?? '', 'free'); ?>>Free</option>
                            <option value="freemium" <?php selected($_REQUEST['filter_license_type'] ?? '', 'freemium'); ?>>Freemium
                            </option>
                            <option value="paid" <?php selected($_REQUEST['filter_license_type'] ?? '', 'paid'); ?>>Paid</option>
                            <option value="unknown" <?php selected($_REQUEST['filter_license_type'] ?? '', 'unknown'); ?>>Unknown
                            </option>
                        </select>
                    </div>
                    <!-- Per Page Dropdown -->
                    <div>
                        <label style="display:block; margin-bottom:5px; font-weight:600;">Items Per Page</label>
                        <select name="per_page" style="width:100%;">
                            <?php
                            $current_pp = intval($_REQUEST['per_page'] ?? 20);
                            foreach ([20, 50, 100, 200] as $opt) {
                                printf('<option value="%d" %s>%d</option>', $opt, selected($current_pp, $opt, false), $opt);
                            }
                            ?>
                        </select>
                    </div>
                </div>
                <div style="margin-top:15px;">
                    <?php submit_button('Apply Filters', 'primary', 'filter', false); ?>
                    <a href="<?php echo esc_url(remove_query_arg(array_merge($filter_names, ['s', 'per_page']))); ?>" class="button"
                        style="margin-left:10px;">Clear All</a>
                </div>
            </div>

            <script type="text/javascript">
                jQuery(document).ready(function ($) {
                    // Initialize Select2 on category dropdown
                    $('#filter_plugin_category').select2({
                        placeholder: 'All Categories',
                        allowClear: true,
                        width: '100%'
                    });
                });
            </script>

            <style>
                /* Custom Select2 styling to match WordPress admin */
                .select2-container--default .select2-selection--single {
                    height: 30px !important;
                    border-color: #8c8f94 !important;
                    border-radius: 4px !important;
                }

                .select2-container--default .select2-selection--single .select2-selection__rendered {
                    line-height: 28px !important;
                    padding-left: 8px !important;
                }

                .select2-container--default .select2-selection--single .select2-selection__arrow {
                    height: 28px !important;
                }

                .select2-container--default .select2-results__option--highlighted[aria-selected] {
                    background-color: #2271b1 !important;
                }
            </style>
            <?php
        }
    }


    /** Override search box */
    public function search_box($text, $input_id)
    {
        return;
    }

    /** Fix: Show no-items row cleanly without overlap */
    public function display_rows_or_placeholder()
    {
        if (empty($this->items)) {
            echo '<style>
                /* Prevent table overlap when no items found */
                .wp-list-table.plugins tbody {
                    display: block;
                    margin-top: 20px;
                }
                
                /* Ensure proper spacing for no-items message */
                .wp-list-table .no-items-row {
                    display: table-row !important;
                }
                
                .wp-list-table .no-items-row td {
                    text-align: center;
                    padding: 40px 30px;
                    background: #fff;
                    border-top: 1px solid #ddd;
                    font-size: 14px;
                    color: #666;
                }
                
                /* Alternative: Add margin to entire table wrapper */
                .tablenav.top ~ .wp-list-table {
                    margin-top: 260px;
                }
            </style>';
            echo '<tr class="no-items-row"><td colspan="' . count($this->get_columns()) . '" class="colspanchange">No plugins found.</td></tr>';
        } else {
            $this->display_rows();
        }
    }
}