<?php
class plugin_comparison_managemnt
{
    public static function plugin_comparison_managemnt()
    {
        add_options_page(
            'Plugin Comparison Management',                                         // Page title
            'Plugin Comparison Management',                                         // Menu title
            'manage_options',                                                       // Capability
            'plugin-comparison-management',                                         // Menu slug
            ['plugin_comparison_managemnt', 'plugin_comparison_management_callback']// Function to display content
        );
        add_management_page(
            'Plugin Comparison JSON Files',              // Page title
            'Plugin Comparison JSON Files',              // Menu title
            'manage_options',                 // Capability
            'plugin-comparison-json-files',              // Menu slug
            ['plugin_comparison_managemnt', 'render_plugin_comparison_json_files_page'],  // Callback
            'dashicons-media-code',           // Icon
            80                                // Position
        );

    }
    public static function render_plugin_comparison_json_files_page()
    {
        // --------------------------------------------------
        // Use SAME approach as working function
        // --------------------------------------------------
        $dir = get_stylesheet_directory() . '/plugins-comparison-data-jsons/';
        $files = glob($dir . '*.json') ?: [];

        // ---------------------------------------------
        // SORT FILES BY NUMBER (1,2,3...10)
        // ---------------------------------------------
        usort($files, function ($a, $b) {
            preg_match('/(\d+)/', basename($a), $matchA);
            preg_match('/(\d+)/', basename($b), $matchB);

            $numA = isset($matchA[1]) ? intval($matchA[1]) : 0;
            $numB = isset($matchB[1]) ? intval($matchB[1]) : 0;

            return $numA <=> $numB;
        });

        $selected_file = isset($_GET['file']) ? sanitize_text_field($_GET['file']) : '';
        $file_path = $dir . $selected_file;

        echo '<div class="wrap"><h1>Plugin Comparison JSON Files</h1><hr/>';

        // ---------------------------------------------
        // FILE LIST VIEW
        // ---------------------------------------------
        if (empty($selected_file)) {
            echo '<h2>Available Files</h2><ul>';

            if (empty($files)) {
                echo '<li><em>No JSON files found.</em></li>';
            } else {
                foreach ($files as $file) {
                    $basename = basename($file);
                    $edit_url = admin_url(
                        'admin.php?page=plugin-comparison-json-files&file=' . urlencode($basename)
                    );

                    echo '<li><a href="' . esc_url($edit_url) . '">' . esc_html($basename) . '</a></li>';
                }
            }

            echo '</ul>';
            echo '<p><em>Select a file to edit.</em></p>';
            echo '</div>';
            return;
        }

        // ---------------------------------------------
        // FILE NOT FOUND
        // ---------------------------------------------
        if (!file_exists($file_path)) {
            echo '<p style="color:red;">File not found.</p>';
            echo '<code>' . esc_html($file_path) . '</code>';
            echo '</div>';
            return;
        }

        // ---------------------------------------------
        // SAVE FILE CONTENT
        // ---------------------------------------------
        if (
            isset($_POST['save_json_file']) &&
            check_admin_referer('save_json_file_action', 'save_json_file_nonce')
        ) {
            $new_content = wp_unslash($_POST['json_content']);
            file_put_contents($file_path, $new_content);

            echo '<div class="updated notice">';
            echo '<p><strong>File saved successfully!</strong></p>';
            echo '</div>';
        }

        // ---------------------------------------------
        // EDIT VIEW
        // ---------------------------------------------
        $content = file_get_contents($file_path);
        $filename = esc_html($selected_file);

        echo '<h2>Editing: ' . $filename . '</h2>';
        echo '<form method="post">';

        wp_nonce_field('save_json_file_action', 'save_json_file_nonce');

        echo '<textarea name="json_content" rows="30" style="width:100%; font-family:monospace;">'
            . esc_textarea($content)
            . '</textarea>';

        echo '<p>';
        echo '<input type="submit" name="save_json_file" class="button button-primary" value="Save File">';
        echo '</p>';

        echo '<p>';
        echo '<a href="' . esc_url(admin_url('admin.php?page=plugin-comparison-json-files')) . '">';
        echo '← Back to File List</a>';
        echo '</p>';

        echo '</form>';
        echo '</div>';

    }
    public static function plugin_comparison_management_callback()
    {
        require get_template_directory() . '/admin/plugin-comparison-management/index.php';
    }
    public static function get_posts_by_plugin_category()
    {
        $category_id = intval($_POST['category_id'] ?? 0);
        $exclude = array_map('intval', $_POST['exclude'] ?? []);

        if (!$category_id) {
            wp_send_json([]);
        }

        $query = new WP_Query([
            'post_type' => 'plugins',
            'posts_per_page' => -1,
            'post__not_in' => $exclude,
            'tax_query' => [
                [
                    'taxonomy' => 'plugin_category',
                    'field' => 'term_id',
                    'terms' => $category_id,
                ],
            ],
        ]);

        $data = [];

        while ($query->have_posts()) {
            $query->the_post();
            $data[] = [
                'id' => get_the_ID(),
                'text' => str_replace(
                    ['&#8211;', ' &amp;'],
                    ['–', '&'],
                    get_the_title(),
                ),
            ];
        }

        wp_reset_postdata();
        wp_send_json($data);
    }
    public static function save_plugin_category_map()
    {
        if (empty($_POST['rows'])) {
            update_option('plugin_comparison_map', [], false);
            wp_send_json_success();
        }

        $rows = [];

        foreach ($_POST['rows'] as $row) {

            if (empty($row['category_id']) || empty($row['posts'])) {
                continue;
            }

            $rows[] = [
                'category_id' => (int) $row['category_id'],
                'posts' => array_map('intval', (array) $row['posts']),
            ];
        }
        self::save_compare_options_in_custom_table();
        update_option('plugin_comparison_map', $rows, false); // ✅ autoload = false
        wp_send_json_success();
    }
    public static function save_compare_options_in_custom_table()
    {
        global $wpdb;

        $table = $wpdb->prefix . 'plugin_comparisons';

        if (empty($_POST['rows']) || !is_array($_POST['rows'])) {
            wp_send_json_success();
        }

        foreach ($_POST['rows'] as $row) {

            /**
             * --------------------------------
             * Validate input
             * --------------------------------
             */
            if (
                empty($row['category_id']) ||
                empty($row['posts']) ||
                !is_array($row['posts'])
            ) {
                continue;
            }

            $category_id = (int) $row['category_id'];

            /**
             * --------------------------------
             * Normalize post IDs
             * --------------------------------
             */
            $post_ids = array_values(
                array_unique(
                    array_map('intval', $row['posts'])
                )
            );

            // 🔒 Allow only 2 or 3 plugins
            if (count($post_ids) < 2 || count($post_ids) > 3) {
                continue;
            }

            /**
             * --------------------------------
             * Resolve plugin slugs & names
             * (order preserved)
             * --------------------------------
             */
            $plugin_slugs = [];
            $plugin_names = [];

            foreach ($post_ids as $post_id) {
                $post = get_post($post_id);
                if ($post && !is_wp_error($post)) {
                    $plugin_slugs[] = $post->post_name;
                    $plugin_names[] = $post->post_title;
                }
            }

            // Ensure valid plugins
            if (count($plugin_slugs) < 2) {
                continue;
            }

            $plugin_slugs_csv = implode(',', $plugin_slugs);
            $plugin_names_csv = implode(',', $plugin_names);

            /**
             * --------------------------------
             * Generate UNIQUE key
             * category + selected IDs
             * --------------------------------
             */
            $plugin_slugs_key = md5(
                $category_id . '|' . implode(',', $post_ids)
            );

            /**
             * --------------------------------
             * Insert (DB UNIQUE will prevent dupes)
             * --------------------------------
             */
            $inserted = $wpdb->insert(
                $table,
                [
                    'plugin_slugs' => $plugin_slugs_csv,
                    'plugin_names' => $plugin_names_csv,
                    'plugin_slugs_key' => $plugin_slugs_key,
                    'plugin_category_id' => $category_id,
                ],
                [
                    '%s',
                    '%s',
                    '%s',
                    '%d',
                ]
            );

            /**
             * --------------------------------
             * Optional: silently ignore duplicates
             * --------------------------------
             */
            if ($inserted === false && $wpdb->last_error) {
                // Duplicate key → ignore
                continue;
            }
        }

        wp_send_json_success();

    }
    public static function creating_json_for_comparison()
    {
        $taxonomy = 'plugin_category';
        $post_type = 'plugins';
        $json_dir = get_stylesheet_directory() . '/plugins-comparison-data-jsons/';

        /**
         * --------------------------------------------------
         * Ensure directory exists
         * --------------------------------------------------
         */
        if (!file_exists($json_dir)) {
            wp_mkdir_p($json_dir);
        }

        /**
         * --------------------------------------------------
         * 1. Get ONLY parent categories
         * --------------------------------------------------
         */
        $parent_categories = get_terms([
            'taxonomy' => $taxonomy,
            'parent' => 0,
            'hide_empty' => true,
        ]);

        if (is_wp_error($parent_categories) || empty($parent_categories)) {
            return;
        }

        /**
         * --------------------------------------------------
         * 2. Loop parent categories
         * --------------------------------------------------
         */
        foreach ($parent_categories as $category) {

            $plugins = get_posts([
                'post_type' => $post_type,
                'posts_per_page' => -1,
                'fields' => 'ids',
                'tax_query' => [
                    [
                        'taxonomy' => $taxonomy,
                        'field' => 'term_id',
                        'terms' => $category->term_id,
                    ],
                ],
            ]);

            if (empty($plugins)) {
                continue;
            }

            $plugin_names = [];
            $plugin_slugs = [];

            foreach ($plugins as $plugin_id) {
                $plugin_names[] = get_the_title($plugin_id);
                $plugin_slugs[] = get_post_field('post_name', $plugin_id);
            }

            $category_data = [
                [
                    'plugin_names' => str_replace(
                        ['&#8211;', ' &amp;'],
                        ['–', '&'],
                        $plugin_names
                    ),
                    'plugin_slugs' => $plugin_slugs,
                    'category_slug' => $category->slug,
                ]
            ];

            /**
             * --------------------------------------------------
             * 3. Write CATEGORY JSON file
             * --------------------------------------------------
             */
            $json_file = $json_dir . $category->slug . '.json';

            $json_data = json_encode(
                $category_data,
                JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
            );

            file_put_contents($json_file, $json_data);
        }

    }
    public static function saving_json_for_comparison()
    {

        $dir = get_stylesheet_directory() . '/plugins-comparison-data-jsons/';

        if (!file_exists($dir)) {
            error_log('❌ JSON directory not found: ' . $dir);
            return false;
        }

        $files = glob($dir . '*.json');
        if (empty($files)) {
            error_log('ℹ️ No JSON files found in directory: ' . $dir);
            return false;
        }

        $option_rows = [];

        // 🔧 Adjust if needed
        $taxonomy = 'plugin_category';
        $post_type = 'plugins';

        /**
         * ---------------------------------------------------------
         * Load existing option to avoid duplicates across executions
         * ---------------------------------------------------------
         */
        $existing_groups = [];

        $existing_option = get_option('plugin_comparison_map', []);
        if (is_array($existing_option)) {
            foreach ($existing_option as $row) {
                if (
                    !empty($row['category_id']) &&
                    !empty($row['posts']) &&
                    is_array($row['posts'])
                ) {
                    $posts = array_map('intval', $row['posts']);
                    sort($posts);

                    $key = (int) $row['category_id'] . '|' . implode(',', $posts);
                    $existing_groups[$key] = true;
                }
            }
        }

        /**
         * -------------------------
         * Process JSON files
         * -------------------------
         */
        foreach ($files as $file_path) {

            $json = file_get_contents($file_path);
            if (!$json) {
                error_log('⚠️ Could not read file: ' . basename($file_path));
                continue;
            }

            $data = json_decode($json, true);
            if (!is_array($data)) {
                error_log('⚠️ Invalid JSON: ' . basename($file_path));
                continue;
            }

            foreach ($data as $group) {

                if (
                    empty($group['category_slug']) ||
                    empty($group['plugin_slugs']) ||
                    !is_array($group['plugin_slugs'])
                ) {
                    continue;
                }

                // 🚫 INVALID if more than 3 plugins
                if (count($group['plugin_slugs']) > 3) {
                    continue;
                }

                // 1️⃣ Get category term
                $term = get_term_by('slug', $group['category_slug'], $taxonomy);
                if (!$term || is_wp_error($term)) {
                    error_log('⚠️ Category not found: ' . $group['category_slug']);
                    continue;
                }

                $post_ids = [];

                // 2️⃣ Resolve plugin posts
                foreach ($group['plugin_slugs'] as $plugin_slug) {
                    $post = get_page_by_path($plugin_slug, OBJECT, $post_type);
                    if ($post && !is_wp_error($post)) {
                        $post_ids[] = (int) $post->ID;
                    }
                }

                // Minimum 2 posts rule
                if (count($post_ids) < 2) {
                    continue;
                }

                // Normalize posts (order-independent)
                $post_ids = array_values(array_unique(array_map('intval', $post_ids)));
                sort($post_ids);

                // Build unique group key
                $group_key = (int) $term->term_id . '|' . implode(',', $post_ids);

                // 3️⃣ Skip if already exists
                if (isset($existing_groups[$group_key])) {
                    continue;
                }

                // Mark as added
                $existing_groups[$group_key] = true;

                // 4️⃣ Add to option rows
                $option_rows[] = [
                    'category_id' => (int) $term->term_id,
                    'posts' => $post_ids,
                ];
            }

            // Optional: delete processed file
            unlink($file_path);
        }

        /**
         * -------------------------
         * Save option
         * -------------------------
         */
        if (!empty($option_rows)) {

            // Merge with existing option to preserve old data
            $final_rows = array_merge($existing_option, $option_rows);

            update_option('plugin_comparison_map', $final_rows, false);

            error_log('✅ Plugin comparison map saved. New groups added: ' . count($option_rows));
        } else {
            error_log('ℹ️ No new comparison groups to save.');
        }

        return true;

    }
    public static function create_json_of_plugins_name_slug()
    {
        $args = [
            'post_type' => 'plugins', // change if needed
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        ];

        $post_ids = get_posts($args);

        $data = [];

        foreach ($post_ids as $post_id) {
            $data[] = [
                'name' => get_the_title($post_id),
                'slug' => get_post_field('post_name', $post_id),
            ];
        }

        // Active theme directory path
        $theme_dir = get_stylesheet_directory();

        // JSON file path
        $file_path =ABSPATH .'/cron-management/plugin_post_name_and_slug.json';

        // Save JSON file
        file_put_contents(
            $file_path,
            json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
        );

        return $file_path;
    }


}
