<?php
class WPS_score_helpers
{
    public static function calculateRMSGrowthDownloaded($data)
    {
        $data = json_decode($data, true);
        // Extract only numeric values
        $values = array_values($data);

        if (count($values) < 2) {
            return 0; // minimum score
        }

        $growthRates = [];

        // Calculate growth between each consecutive value
        for ($i = 1; $i < count($values); $i++) {
            $prev = $values[$i - 1];
            $current = $values[$i];

            if ($prev > 0) {
                $growth = ($current - $prev) / $prev;
                $growthRates[] = $growth;
            }
        }

        if (empty($growthRates)) {
            return 1;
        }

        // RMS = sqrt( sum(growth²) / n )
        $sumSquares = 0;
        foreach ($growthRates as $g) {
            $sumSquares += ($g * $g);
        }

        $rmsGrowth = sqrt($sumSquares / count($growthRates));

        // ------------------------------
        // SCALE + MAP TO 1–10 RANGE
        // ------------------------------

        $scaled = $rmsGrowth * 10000;  // scale tiny numbers up

        // clamp to 1 – 10
        $score = max(1, min(10, $scaled));

        return $score;
    }
    public static function calculate_support_responsiveness($support_threads, $support_threads_resolved)
    {
        if ($support_threads <= 0) {
            return 9;
        }

        // Calculate resolution ratio
        $ratio = $support_threads_resolved / $support_threads;

        // Scale ratio to 1-10
        // Fully resolved (100%) = 10, nothing resolved = 1
        $score = 1 + ($ratio * 9);

        // Ensure score is between 1 and 10
        $score = max(1, min(10, round($score, 1)));

        return $score;
    }
    public static function get_latest_wordpress_version()
    {
        // WordPress API URL for core versions
        $url = 'https://api.wordpress.org/core/version-check/1.7/';

        // Get the JSON response from the API
        $response = wp_remote_get($url);

        if (is_wp_error($response)) {
            return 'Unable to fetch version';
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (isset($data['offers'][0]['version'])) {
            return $data['offers'][0]['version'];
        }

        return 'Version not found';
    }
    public static function calculate_rating_score_of_ten($rating)
    {

    }
    public static function calculateMetricStats($values)
    {
        if (empty($values)) {
            return [
                'min' => 0,
                'max' => 0,
                'avg' => 0,
            ];
        }

        return [
            'min' => min($values),
            'max' => max($values),
            'avg' => array_sum($values) / count($values),
        ];
    }
    public static function count_supported_php_versions($min_php)
    {
        $php_major_versions = [
            '5.0',
            '5.1',
            '5.2',
            '5.3',
            '5.4',
            '5.5',
            '5.6',
            '7.0',
            '7.1',
            '7.2',
            '7.3',
            '7.4',
            '8.0',
            '8.1',
            '8.2',
            '8.3'
        ];

        if (empty($min_php)) {
            return count($php_major_versions);
        }

        $count = 0;

        foreach ($php_major_versions as $version) {
            if (version_compare($version, $min_php, '>=')) {
                $count++;
            }
        }

        return $count;
    }
    public static function count_dynamic_keys(array $map, array $values): array
    {
        // Initialize result with all keys as 0
        $result = array_fill_keys(array_keys($map), 0);

        // Loop through values and increment count
        foreach ($values as $v) {
            if (array_key_exists($v, $result)) {
                $result[$v]++;
            }
        }

        return $result;
    }
    public static function get_floating_score_by_key_value_count_compare_with_key(array $counts, string|int $key): float
    {
        if (!array_key_exists($key, $counts)) {
            return 1.0; // lowest score
        }

        $values = array_values($counts);
        $sum = array_sum($values);

        if ($sum == 0) {
            return 5.0; // neutral fallback
        }

        $value = $counts[$key];

        // Convert to percentage (0–1)
        $p = $value / $sum;

        /**
         * 📌 STATISTICAL AMPLIFICATION
         * log scaling + cubic shaping → stronger jumps, less flattening
         *
         * - log1p() grows fast in high values
         * - pow(p, 3/2) spreads distribution
         */
        $boost = (log1p($value) / log1p(max($values))) * 0.6
            + pow($p, 1.5) * 0.4;

        // Combine (weights ensure stable result)
        $score = 1 + 9 * $boost;

        return round(max(1, min(9.9, $score)), 2);
    }

    public static function countValuesOfElementOn1DArray(array $items): array
    {
        $result = [];

        foreach ($items as $value) {
            if (!isset($result[$value])) {
                $result[$value] = 0;
            }
            $result[$value]++;
        }

        return $result;
    }
    public static function ensure_zero_key(array $arr): array
    {
        if (!array_key_exists(0, $arr)) {
            $arr[0] = 0;
        }
        ksort($arr); // optional: sort keys ascending
        return $arr;
    }
    public static function metric_with_avg_max_min($min, $max, $avg, $value, $reverse = false)
    {
        if ($max == $min) {
            return 5.00;
        }

        // Normalize value between 0–1
        $norm_value = ($value - $min) / ($max - $min);
        $norm_avg = ($avg - $min) / ($max - $min);

        // Standard z-score difference (spread stronger)
        $diff = $norm_value - $norm_avg;

        /**
         * 📌 Non-linear amplification
         * - diff^3 makes high performers jump higher
         * - keeps mid-values stable
         */
        $amplified = $diff * 4 + pow($diff, 3) * 3;

        /**
         * Base score is 5 (average)
         * + amplified variation
         */
        $score = 5 + ($amplified * 5); // final multiplier controls spread strength

        // Reverse scoring (lower = better)
        if ($reverse) {
            $score = 11 - $score;
        }

        // Clamp and return
        return round(max(3.5, min(9.9, $score)), 2);
    }
    public static function percent_to_rating_10($percent, $maxRating = 10)
    {
        // Ensure value is numeric and within range
        $percent = max(0, min(100, floatval($percent)));

        // Convert percentage to rating out of 10
        $rating = ($percent / 100) * $maxRating;

        // Round to 1 decimal (e.g., 8.7)
        return round($rating, 1);
    }
    public static function average_floats(float ...$numbers): float
    {
        if (empty($numbers)) {
            return 0.00;
        }

        $avg = array_sum($numbers) / count($numbers);
        return round($avg, 2);
    }
    public static function score_to_performance_level_enum(float $score): int
    {
        // Ensure score stays in valid range
        $score = max(1, min(10, $score));

        // Convert 1–10 → 1–3
        $index = (int) ceil(($score / 10) * 3);

        // Ensure 1 ≤ index ≤ 3
        return max(1, min(3, $index));
    }
    public static function get_the_reputation_score($author_profile)
    {
        global $wpdb;

        $sql = $wpdb->prepare(
            "
        SELECT 
            AVG(overall_wpscore) AS wps_developer_reputation_score
        FROM 
            {$wpdb->prefix}plugins_core_information
        WHERE 
            author_profile = %s
        ",
            $author_profile
        );

        $row = $wpdb->get_row($sql);
        //return $sql;
        return $row->wps_developer_reputation_score;
    }
    public static function weighted_score_to_sentiment(array $scoreCounts): int
    {
        if (empty($scoreCounts)) {
            return 2; // Neutral fallback
        }

        // Calculate total count
        $totalCount = array_sum($scoreCounts);
        if ($totalCount == 0) {
            return 2; // Neutral fallback
        }

        // Calculate weighted average
        $weightedSum = 0;
        foreach ($scoreCounts as $score => $count) {
            // Ensure score is numeric
            $score = floatval($score);
            $weightedSum += $score * $count;
        }

        $weightedAvg = $weightedSum / $totalCount;

        /**
         * Map weighted average (1–5 scale) to sentiment enum 1–3
         * Example:
         * 1.0–2.33 → Negative (1)
         * 2.34–3.66 → Neutral (2)
         * 3.67–5.0 → Positive (3)
         */
        if ($weightedAvg <= 2.33) {
            return 1; // Negative
        } elseif ($weightedAvg <= 3.66) {
            return 2; // Neutral
        } else {
            return 3; // Positive
        }
    }
    public static function support_quality_enum(int $supportThreads, int $supportThreadsResolved): int
    {
        // Avoid division by zero
        if ($supportThreads <= 0) {
            return 4; // Poor by default if no threads exist
        }

        // Calculate resolution ratio
        $ratio = $supportThreadsResolved / $supportThreads;

        /**
         * Map ratio to 1–4 enum:
         * ratio >= 0.9 → Excellent (1)
         * ratio >= 0.75 → Good (2)
         * ratio >= 0.5 → Fair (3)
         * ratio < 0.5 → Poor (4)
         */
        if ($ratio >= 0.8) {
            return 1; // Excellent
        } elseif ($ratio >= 0.65) {
            return 2; // Good
        } elseif ($ratio >= 0.5) {
            return 3; // Fair
        } else {
            return 4; // Poor
        }
    }
    public static function weighted_score_1_to_10(array $scores, array $weights): float
    {
        // If empty input, return neutral mid score
        if (empty($scores) || empty($weights)) {
            return 5.0;
        }

        // Ensure score and weight count match
        if (count($scores) !== count($weights)) {
            return 5.0;
        }

        // Normalize weights (convert 7, 3, 20 into 0.07, 0.03, 0.20)
        $weight_sum = array_sum($weights);

        if ($weight_sum <= 0) {
            return 5.0;
        }

        $normalized_weights = array_map(function ($w) use ($weight_sum) {
            return $w / $weight_sum;
        }, $weights);

        // Calculate weighted sum (scores expected to be 1–10 scale)
        $weighted_total = 0;

        foreach ($scores as $i => $score) {
            // Auto-fix score if value like 7.69, 9.9 or any float
            $score = max(1, min(10, floatval($score)));

            $weighted_total += $score * $normalized_weights[$i];
        }

        // Final score between 1 and 10
        return round(max(1, min(10, $weighted_total)), 2);
    }
    public static function get_weight_by_attribute(array $data, string $attribute): ?float
    {
        foreach ($data as $group) {              // core_performance, business_fit, etc.
            foreach ($group as $item) {          // each item inside group
                if (isset($item['attribute']) && $item['attribute'] === $attribute) {
                    return $item['weight'];      // return weight immediately
                }
            }
        }

        return null; // attribute not found
    }

}