<?php
// Load WordPress environment safely
require_once dirname(__FILE__) . '/index.php';

class WpPluginSizeCalculation
{

    public function __construct()
    {

        // Check & acquire lock
        if (! wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->wpPluginSizeCalculationFunction();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — updates active_installs in cron_scraped_plugins_log
     */
    private function wpPluginSizeCalculationFunction()
    {
        global $wpdb;

        $tbl_plugins_core_information = $wpdb->prefix . "plugins_core_information";

        update_option('wp_plugin_size_getting_job_last_run', current_time('mysql'), false);

        // ✅ Fetch pages that need visiting
        $plugins_core_information = $wpdb->get_results(
            "SELECT id,last_updated,package_size_kb,download_link
        FROM {$tbl_plugins_core_information}
        WHERE
        updated_at IS NULL
        OR DATEDIFF(NOW(), updated_at) > 7
        OR package_size_kb = 0"
        );

        if (empty($plugins_core_information)) {
            echo "No Plugins found.\n";
            return;
        }

        foreach ($plugins_core_information as $single_plugin_information) {
            $size_in_kb = $this->get_size_in_kb($single_plugin_information->download_link);
            // ✅ Mark this page as visited
            $wpdb->update(
                $tbl_plugins_core_information,
                [
                    'package_size_kb' => $size_in_kb,
                ],
                [
                    'id' => $single_plugin_information->id,
                ],
                ['%d'],
                ['%d']
            );
        }
        update_option('wp_plugin_size_getting_job_last_run', current_time('mysql'), false);
    }
    private function get_remote_file_size($url)
    {
        $ch = curl_init($url);

        // Use HEAD request to avoid downloading the whole file
        curl_setopt($ch, CURLOPT_NOBODY, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 5);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);

        curl_exec($ch);

        // Get file size from headers
        $filesize = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);

        curl_close($ch);

        if ($filesize > 0) {
            return $filesize; // in bytes
        }

        return false; // couldn't determine size
    }
    private function get_size_in_kb($url)
    {

        $size = $this->get_remote_file_size($url);

        if ($size !== false && $size > 0) {
            // Convert bytes → KB
            $size_in_kb = $size / 1024;
            return round($size_in_kb, 2); // 2 decimals is fine for KB
        } else {
            error_log("Unable to get the size: " . $url);
            return 0;
        }
    }
}

// Instantiate the class
new WpPluginSizeCalculation();
