<?php
require_once dirname(__FILE__) . '/index.php';
class WpPluginDirectorySinglePluginSeeding
{

    public function __construct()
    {

        // Check & acquire lock
        if (! wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->seedingWPOrgPluginJSON();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    private function seedingWPOrgPluginJSON()
    {
        global $wpdb;

        $table_cron_page_visits = $wpdb->prefix . "cron_page_visits_log";     // Make sure to use the prefix for table names
        $cron_scraped_plugins   = $wpdb->prefix . "cron_scraped_plugins_log"; // Use table prefix for scraped plugins

        // Get all the pages that haven't been visited
        $cron_page_visits = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_cron_page_visits WHERE visited = 0"
        ));

        // Loop through each cron page visit entry
        foreach ($cron_page_visits as $single_object) {
            sleep(3);
            // Fetch plugin data from the endpoint
            $response_data = $this->apiGet($single_object->endpoint);

            // If the plugins data exists in the response
            if (isset($response_data["plugins"]) && ! empty($response_data["plugins"])) {
                foreach ($response_data["plugins"] as $single_plugin_information) {
                    try {
                        // ✅ Check if icon exists before accessing
                        $plugin_icon = isset($single_plugin_information['icons']['1x'])
                            ? $single_plugin_information['icons']['1x']
                            : null;

                        $plugin_slug = $single_plugin_information['slug'] ?? null;

                        // ✅ Skip if slug is empty
                        if (empty($plugin_slug)) {
                            continue;
                        }

                        // ✅ Fast existence check (SELECT 1)
                        $exists = $wpdb->get_var(
                            $wpdb->prepare(
                                "SELECT 1 FROM {$cron_scraped_plugins} WHERE plugin_slug = %s LIMIT 1",
                                $plugin_slug
                            )
                        );

                        if ($exists) {
                            // Already exists → skip insert
                            continue;
                        }

                        // ✅ Prepare insert data
                        $insert_data = [
                            'plugin_slug' => $plugin_slug,
                            'plugin_name' => $single_plugin_information['name'] ?? null,
                            'plugin_icon' => $plugin_icon,
                            'page_number' => $single_object->page_number ?? null,
                        ];

                        // ✅ Define format (order must match $insert_data)
                        $insert_format = [
                            '%s', // plugin_slug
                            '%s', // plugin_name
                            '%s', // plugin_icon
                            '%d', // page_number
                        ];

                        // ✅ Perform the insert
                        $result = $wpdb->insert($cron_scraped_plugins, $insert_data, $insert_format);

                        if ($result === false) {
                            error_log('Database insert failed: ' . $wpdb->last_error);
                            continue;
                        }

                    } catch (Exception $e) {
                        error_log('Error inserting plugin data: ' . $e->getMessage());
                        continue;
                    }

                }
            }

            // After processing the plugins, mark this page as visited
            $wpdb->update(
                $table_cron_page_visits,
                [
                    'visited' => 1, // Mark as visited
                ],
                ['id' => $single_object->id], // WHERE condition: only the current record
                ['%d'],                       // Format for visited field
                ['%d']                        // Format for the id field
            );
        }
    }

    private function apiGet($url)
    {
        $curl = curl_init();

        // Set cURL options
        curl_setopt_array($curl, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING       => '',
            CURLOPT_MAXREDIRS      => 10,
            CURLOPT_TIMEOUT        => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST  => 'GET',
        ]);

        // Execute the request
        $response = curl_exec($curl);

        // Check if there was a cURL error
        if (curl_errno($curl)) {
            throw new Exception('cURL error: ' . curl_error($curl));
        }

        // Get the status code of the response
        $statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        // Close cURL session
        curl_close($curl);

        // Check if the response is successful
        if ($statusCode !== 200) {
            throw new Exception("HTTP $statusCode: $response");
        }

        // Return the decoded JSON response
        return json_decode($response, true);
    }
}

// Instantiate the class to trigger the process
new WpPluginDirectorySinglePluginSeeding();
