<?php
// Load WordPress environment safely
require_once dirname(__FILE__) . '/index.php';

class WpCreateTaxonomyFromJSON
{

    public function __construct()
    {

        // Check & acquire lock
        if (! wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->wpCreateTaxonomyFromJSON();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — updates active_installs in cron_scraped_plugins_log
     */
    private function wpCreateTaxonomyFromJSON()
    {
        $taxonomy         = 'plugin_category';
        $master_json_path = plugin_dir_path(__FILE__) . 'possible_category.json';

        if (! file_exists($master_json_path)) {
            error_log("❌ Master taxonomy JSON file not found at: {$master_json_path}");
            return ['success' => false, 'message' => 'JSON file not found'];
        }

        $master_data = json_decode(file_get_contents($master_json_path), true);
        if (! isset($master_data['categories']) || ! is_array($master_data['categories'])) {
            error_log("❌ Invalid master taxonomy JSON structure.");
            return ['success' => false, 'message' => 'Invalid JSON structure'];
        }

        $stats = [
            'categories_created'     => 0,
            'categories_existing'    => 0,
            'subcategories_created'  => 0,
            'subcategories_existing' => 0,
        ];

        foreach ($master_data['categories'] as $category) {
            $main_name = $category['name'];

            // ✅ CRITICAL: Generate slug using wps_create_slug (same function used for AI lookup)
            $main_slug = wps_create_slug($main_name);

            // Check if term exists by slug
            $existing_term = get_term_by('slug', $main_slug, $taxonomy);

            if ($existing_term) {
                // Term exists - update if name changed
                if ($existing_term->name !== $main_name) {
                    wp_update_term($existing_term->term_id, $taxonomy, [
                        'name' => $main_name,
                    ]);
                }
                $main_term_id = $existing_term->term_id;
                $stats['categories_existing']++;
            } else {
                // Create new term
                $result = wp_insert_term($main_name, $taxonomy, [
                    'slug' => $main_slug,
                ]);

                if (is_wp_error($result)) {
                    error_log("❌ Failed to create category '{$main_name}': " . $result->get_error_message());
                    continue;
                }

                $main_term_id = $result['term_id'];
                $stats['categories_created']++;
            }

            // ✅ Create/update subcategories
            if (! empty($category['subcategories']) && is_array($category['subcategories'])) {
                foreach ($category['subcategories'] as $subcat) {
                    $sub_name = $subcat['name'];

                    // ✅ CRITICAL: Generate slug using wps_create_slug
                    $sub_slug = wps_create_slug($sub_name);

                    // Check if subcategory exists by slug
                    $existing_subterm = get_term_by('slug', $sub_slug, $taxonomy);

                    if ($existing_subterm) {
                        // Update if name or parent changed
                        $needs_update = false;
                        $update_args  = [];

                        if ($existing_subterm->name !== $sub_name) {
                            $update_args['name'] = $sub_name;
                            $needs_update        = true;
                        }

                        if ($existing_subterm->parent !== $main_term_id) {
                            $update_args['parent'] = $main_term_id;
                            $needs_update          = true;
                        }

                        if ($needs_update) {
                            wp_update_term($existing_subterm->term_id, $taxonomy, $update_args);
                        }

                        $stats['subcategories_existing']++;
                    } else {
                        // Create new subcategory
                        $result = wp_insert_term($sub_name, $taxonomy, [
                            'slug'   => $sub_slug,
                            'parent' => $main_term_id,
                        ]);

                        if (is_wp_error($result)) {
                            error_log("❌ Failed to create subcategory '{$sub_name}': " . $result->get_error_message());
                            continue;
                        }

                        $stats['subcategories_created']++;
                    }
                }
            }
        }

        $message = sprintf(
            "✅ Sync complete! Categories: %d created, %d existing | Subcategories: %d created, %d existing",
            $stats['categories_created'],
            $stats['categories_existing'],
            $stats['subcategories_created'],
            $stats['subcategories_existing']
        );

        error_log($message);
        echo json_encode(['success' => true, 'message' => $message, 'stats' => $stats]);
        update_option('wp_create_taxonomy_from_json_last_run', current_time('mysql'), false);
    }

}

// Instantiate the class
new WpCreateTaxonomyFromJSON();
