<?php
require_once dirname(__FILE__) . '/index.php';

class WpPluginAIComparisonGeneration
{

    public function __construct()
    {

        // Check & acquire lock
        if (!wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->run();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — generates AI comparison JSON
     */
    public function run()
    {
        global $wpdb;

        $tbl_plugin_comparisons = $wpdb->prefix . 'plugin_comparisons';

        // Job start time
        update_option('wp_plugin_ai_comparison_job_last_run', current_time('mysql'), false);

        $plugin_comparisons = $wpdb->get_results("
            SELECT *
            FROM {$tbl_plugin_comparisons}
            WHERE is_ai_comparison_json_done = 0
            AND is_post_created = 0
            LIMIT 50
        ");

        if (empty($plugin_comparisons)) {
            return;
        }

        foreach ($plugin_comparisons as $row) {

            // Prepare plugin input
            $plugins = [];
            $slugs = array_map('trim', explode(',', $row->plugin_slugs));
            $names = array_map('trim', explode(',', $row->plugin_names));

            foreach ($slugs as $i => $slug) {
                $plugins[] = [
                    'slug' => $slug,
                    'name' => html_entity_decode($names[$i] ?? '', ENT_QUOTES | ENT_HTML5),
                ];
            }

            // Call OpenAI
            $comparison_json = $this->get_open_ai_filter_data($plugins, $row->id);

            //dd($comparison_json);
            //exit;
            if (!$comparison_json) {
                continue;
            }
            $comparison_json_data = json_decode($comparison_json, true);

            // Save result
            $wpdb->update(
                $tbl_plugin_comparisons,
                [
                    'plugin_comparison' => $comparison_json,
                    'comparison_title' => $comparison_json_data['comparison_title'],
                    'is_ai_comparison_json_done' => 1,
                ],
                ['id' => $row->id],
                ['%s', '%s', '%d'],
                ['%d']
            );
        }

        // Job end time
        update_option('wp_plugin_ai_comparison_job_last_run', current_time('mysql'), false);
    }

    /**
     * Output structure definition
     */
    public function get_output_structure()
    {
        return <<<JSON
            {
            "comparison_title": "",
            "plugins": [
                {
                "name": "",
                "slug": ""
                }
            ],
            "overview": "",
            "feature_table": {
                "rows": [
                {
                    "feature": "",
                    "plugin-slug-1": {
                    "is_available": null,
                    "note": ""
                    },
                    "plugin-slug-2": {
                    "is_available": null,
                    "note": ""
                    },
                    "plugin-slug-3": {
                    "is_available": null,
                    "note": ""
                    }
                }
                ],
                "verdict": ""
            },
            "ease_of_use": {
                "content": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "complexity": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "comparison_level": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "verdict": ""
            },
            "performance": {
                "content": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "speed": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "comparison_level": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "verdict": ""
            },
            "pricing": {
                "content": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "comparison_level": {
                "plugin-slug-1": {
                    "pricing": "",
                    "note": ""
                },
                "plugin-slug-2": {
                    "pricing": "",
                    "note": ""
                },
                "plugin-slug-3": {
                    "pricing": "",
                    "note": ""
                }
                },
                "recommendation": ""
            },
            "use_case": {
                "content": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "comparison_reason": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                },
                "verdict": ""
            },
            "pros_cons": {
                "content": {
                "plugin-slug-1": {
                    "pros": [],
                    "cons": []
                },
                "plugin-slug-2": {
                    "pros": [],
                    "cons": []
                },
                "plugin-slug-3": {
                    "pros": [],
                    "cons": []
                }
                },
                "comparison_level": {
                "plugin-slug-1": "",
                "plugin-slug-2": "",
                "plugin-slug-3": ""
                }
            },
            "final_verdict": {
                "plugin-slug-1": {
                "verdict": "",
                "comparison_level": ""
                },
                "plugin-slug-2": {
                "verdict": "",
                "comparison_level": ""
                },
                "plugin-slug-3": {
                "verdict": "",
                "comparison_level": ""
                }
            },
            "faq": [
                {
                "question": "",
                "answer": ""
                }
            ]
            }
            JSON;
    }

    /**
     * OpenAI API call
     */
    public function get_open_ai_filter_data($plugins, $id)
    {
        try {
            $OPENAI_API_KEY = YOUR_OPENAI_API_KEY;

            $prompt = "You are a WordPress plugin comparison generator.
            STRICT RULES (MANDATORY):
            - Output ONLY valid JSON
            - No markdown
            - No comments
            - No explanations
            - No trailing commas
            - Match the JSON structure EXACTLY
            - Use plugin SLUGS as object keys
            - Do NOT invent features
            - If data is unknown, use null or empty string

            FEATURE AVAILABILITY:
            - 1 = available
            - 0 = not available
            - null = unclear or partial

            EASE OF USE (complexity values ONLY):
            - simple
            - moderate
            - complex

            PERFORMANCE (speed values ONLY):
            - fast
            - moderate
            - slow

            COMPARISON LEVEL VALUES (ONLY):
            - win
            - high
            - moderate

            GLOBAL RULES:
            - EXACTLY ONE plugin must have comparison_level = \"win\" in final_verdict
            - Other plugins must be \"high\" or \"moderate\"
            - comparison_level must reflect RELATIVE strength per section

            INPUT PLUGINS:
            " . json_encode($plugins, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "

            RETURN JSON EXACTLY MATCHING THIS STRUCTURE:
            " . $this->get_output_structure();

            $payload = [
                "model" => "gpt-4.1-mini",
                "messages" => [
                    ["role" => "user", "content" => $prompt],
                ],
                "temperature" => 0.3,
            ];

            $ch = curl_init("https://api.openai.com/v1/chat/completions");
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_HTTPHEADER => [
                    "Content-Type: application/json",
                    "Authorization: Bearer {$OPENAI_API_KEY}",
                ],
                CURLOPT_POSTFIELDS => json_encode($payload),
            ]);

            $response = curl_exec($ch);

            if (curl_errno($ch)) {
                throw new Exception(curl_error($ch));
            }

            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($http_code < 200 || $http_code >= 300) {
                throw new Exception("OpenAI HTTP {$http_code}: {$response}");
            }

            $data = json_decode($response, true);
            $content = $data['choices'][0]['message']['content'] ?? '';

            // Validate JSON
            json_decode($content);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid JSON returned by OpenAI');
            }

            return $content;

        } catch (Exception $e) {
            error_log('OpenAI Error: ' . $e->getMessage());
            $this->save_error($id, $e->getMessage());
            return false;
        }
    }

    /**
     * Save OpenAI error
     */
    public function save_error($id, $error)
    {
        global $wpdb;
        $tbl_plugin_comparisons = $wpdb->prefix . 'plugin_comparisons';

        $wpdb->update(
            $tbl_plugin_comparisons,
            ['error_log_open_ai' => $error],
            ['id' => $id],
            ['%s'],
            ['%d']
        );
    }
}

// Run
new WpPluginAIComparisonGeneration();
