<?php
require_once dirname(__FILE__) . '/index.php';

class WpPluginAIWizardDataManagement
{
    public function __construct()
    {
        // Check & acquire lock
        if (!wp_job_lock()) {
            echo "already running";
            return;
        }

        try {
            // Call core function ONLY after lock
            $this->pluginWizardDataGeneration();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — generates wizard feature → plugin dataset
     */
    private function pluginWizardDataGeneration()
    {
        update_option(
            'wp_plugin_ai_wizard_data_management_job_last_run',
            current_time('mysql'),
            false
        );

        // Load input data
        $plugins = $this->loadPlugins();
        $features = multistep_plugin_select_wizard::features_need_in_site();
        $builders = multistep_plugin_select_wizard::builder_theme_using();
        $websites = multistep_plugin_select_wizard::type_website_building();

        if (empty($plugins)) {
            echo "No plugins found.\n";
            return;
        }

        // Generate AI dataset
        $json = $this->generateWizardDataset(
            $plugins,
            $features,
            $builders,
            $websites
        );

        if (!$json) {
            echo "Wizard dataset generation failed.\n";
            return;
        }

        // Save final JSON
        $this->saveWizardDataset($json);

        update_option(
            'wp_plugin_ai_wizard_data_management_job_last_run',
            current_time('mysql'),
            false
        );
    }

    /**
     * Load plugin name & slug list
     */
    private function loadPlugins()
    {
        $file = dirname(__FILE__) . '/plugin_post_name_and_slug.json';

        if (!file_exists($file)) {
            return [];
        }

        $data = json_decode(file_get_contents($file), true);

        return is_array($data) ? $data : [];
    }

    /**
     * Output structure reference (AI must follow)
     */
    private function get_output_structure()
    {
        return <<<JSON
        {
        "features": [
            {
            "feature": {
                "name": "SEO",
                "feature_slug": "seo",
                "plugins": [
                {
                    "name": "",
                    "slug": "",
                    "builder_support": [
                    "gutenberg-default"
                    ],
                    "website_support": [
                    "blog"
                    ]
                }
                ]
            }
            }
        ]
        }
        JSON;
    }

    /**
     * Call OpenAI to generate wizard dataset
     */
    private function generateWizardDataset(
        array $plugins,
        array $features,
        array $builders,
        array $websites
    ) {
        try {
            $OPENAI_API_KEY = YOUR_OPENAI_API_KEY;
            $API_URL = "https://api.openai.com/v1/chat/completions";

            $prompt = <<<EOT
            You are an expert WordPress Plugin Wizard Data Engine.

            INPUT DATA:

            PLUGINS (name + slug):
            {$this->json($plugins)}

            FEATURES (feature_slug → feature_name):
            {$this->json($features)}

            BUILDERS (builder_slug → builder_name):
            {$this->json($builders)}

            WEBSITE TYPES (website_slug → website_name):
            {$this->json($websites)}

            TASK:

            - For EACH feature slug, create a feature object
            - Assign ALL relevant plugins to that feature
            - Use plugin name and slug as the strongest signal
            - Use feature SLUGS for matching
            - Determine builder_support using builder slugs
            - Determine website_support using website slugs
            - A plugin MAY appear under MULTIPLE features
            - Do NOT invent plugins, features, builders, or websites
            - Omit features with zero plugins

            SLUG RULES:
            - lowercase
            - hyphenated
            - slugs only (no names)

            RETURN JSON EXACTLY MATCHING THIS STRUCTURE:
            {$this->get_output_structure()}

            OUTPUT RULES:
            - JSON only
            - No markdown
            - No explanations
             --------------------------------------------------
            ### FINAL INSTRUCTION
            --------------------------------------------------
            Return ONLY the JSON object. No explanation. No extra text.
            EOT;
            // -----------------------------------
            // API PAYLOAD (same as your old code)
            // -----------------------------------
            $payload = [
                "model" => "gpt-4o",
                "messages" => [
                    ["role" => "user", "content" => $prompt],
                ],
                "temperature" => 0.4,
                "response_format" => ["type" => "text"],
            ];

            // -----------------------------------
            // CURL REQUEST
            // -----------------------------------
            $ch = curl_init($API_URL);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_HTTPHEADER => [
                    "Content-Type: application/json",
                    "Authorization: Bearer {$OPENAI_API_KEY}",
                ],
                CURLOPT_POSTFIELDS => json_encode($payload),
            ]);

            $response = curl_exec($ch);

            if (curl_errno($ch)) {
                throw new Exception('cURL Error: ' . curl_error($ch));
            }

            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($http_code < 200 || $http_code >= 300) {
                throw new Exception("API request failed: {$response}");
            }

            // -----------------------------------
            // PARSE RESPONSE (same regex logic)
            // -----------------------------------
            $data = json_decode($response, true);
            $content = $data['choices'][0]['message']['content'] ?? '';

            if (preg_match('/\{.*\}/s', $content, $match)) {
                $content = $match[0];
            }

            $json_output = json_decode($content, true);

            if (!$json_output) {
                throw new Exception('Failed to parse AI profile JSON.');
            }

            return json_encode($json_output, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

        } catch (Exception $e) {
            error_log('Wizard AI Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Save wizard dataset JSON to disk
     */
    private function saveWizardDataset(string $json)
    {
        $dir = ABSPATH . 'cron-management/';
        $file = $dir . 'multistep-wizard-data-set.json';

        // Ensure directory exists
        if (!file_exists($dir)) {
            wp_mkdir_p($dir);
        }

        // Write file
        file_put_contents($file, $json, LOCK_EX);

        // Ensure file permission is 0644
        if (file_exists($file)) {
            $current_perms = substr(sprintf('%o', fileperms($file)), -4);
            if ($current_perms !== '0644') {
                @chmod($file, 0644);
            }
        }
    }

    /**
     * JSON helper
     */
    private function json($data)
    {
        return json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    }
}

// Instantiate the class
new WpPluginAIWizardDataManagement();
