<script>
    // Global suggestions storage
    let globalSuggestions = [];
    let suggestionsLoaded = false;

    function isCategoryPage() {
        return document.getElementById("term_name") !== null;
    }

    // Load suggestions on page load
    document.addEventListener("DOMContentLoaded", function() {
        const isCategory = isCategoryPage();

        if (isCategory) {
            const termEl = document.getElementById("term_name");
            const categoryText = termEl ? termEl.innerText.trim() : "";

            const formData = new FormData();
            formData.append("action", "get_plugin_category_suggestions");
            formData.append("category", categoryText);

            fetch("<?php echo admin_url('admin-ajax.php'); ?>", {
                    method: "POST",
                    body: formData
                })
                .then(res => res.json())
                .then(data => {
                    globalSuggestions = data || [];
                    suggestionsLoaded = true;
                })
                .catch(err => {
                    console.error("Error loading suggestions:", err);
                });
        } else {
            fetch("<?php echo get_template_directory_uri(); ?>/plugins-suggestions.json")
                .then(res => res.json())
                .then(data => {
                    globalSuggestions = data || [];
                    suggestionsLoaded = true;
                })
                .catch(err => {
                    console.error("Error loading suggestions:", err);
                });
        }

        // Initialize search boxes after a short delay
        setTimeout(() => {
            initSearch("searchInput", "suggestionBox"); // Desktop
            initSearch("searchInputM", "suggestionBoxM"); // Mobile
        }, 100);
    });

    function initSearch(searchInputId, suggestionBoxId) {
        const searchInput = document.getElementById(searchInputId);
        const suggestionBox = document.getElementById(suggestionBoxId);

        if (!searchInput || !suggestionBox) return;

        let searchTimer = null;
        let isCalculating = false;

        /* ======= ENHANCED FUZZY MATCH FUNCTION ======= */
        function fuzzyMatch(str, query) {
            str = str.toLowerCase();
            query = query.toLowerCase();

            const words = query.split(/\s+/).filter(Boolean);

            return words.every(word => {
                const strWords = str.split(/\s+/);
                const startsWithMatch = strWords.some(w => w.startsWith(word));
                if (startsWithMatch) return true;

                if (word.length <= 2) return str.includes(word);

                const partialLength = Math.ceil(word.length * 0.6);
                return str.includes(word.slice(0, partialLength));
            });
        }

        /* ======= ENHANCED PRIORITY SCORE FUNCTION ======= */
        function getPriorityScore(item, query) {
            if (!query) return 0;

            const q = query.toLowerCase().trim();
            const title = (item.title || "").toLowerCase();
            const text = (item.text || "").toLowerCase();

            let score = 0;

            const queryWords = q.split(/\s+/).filter(Boolean);
            const titleWords = title.split(/\s+/).filter(Boolean);
            const textWords = text.split(/\s+/).filter(Boolean);

            if (title.includes(q)) score += 500;
            if (text.includes(q)) score += 200;

            const allWordsInTitle = queryWords.every(w => title.includes(w));
            const allWordsInText = queryWords.every(w => text.includes(w));

            if (allWordsInTitle) score += 350;
            else if (allWordsInText) score += 150;

            if (queryWords.length > 1) {
                const joinedTitle = titleWords.join(" ");
                queryWords.forEach((w, i) => {
                    if (!queryWords[i + 1]) return;
                    const a = joinedTitle.indexOf(w);
                    const b = joinedTitle.indexOf(queryWords[i + 1]);
                    if (a !== -1 && b !== -1 && Math.abs(a - b) < 15) {
                        score += 150;
                    }
                });
            }

            const orderedMatch = queryWords.every((w, i) =>
                titleWords[i] && titleWords[i].startsWith(w)
            );
            if (orderedMatch) score += 300;

            queryWords.forEach(qWord => {
                titleWords.forEach((tWord, i) => {
                    if (tWord.startsWith(qWord)) {
                        score += qWord.length > 4 ? 120 : 30;
                        if (i === 0) score += 40;
                    }
                });

                textWords.forEach(tWord => {
                    if (tWord.startsWith(qWord)) {
                        score += qWord.length > 4 ? 60 : 15;
                    }
                });
            });

            if (fuzzyMatch(title, q)) score += 30;
            if (fuzzyMatch(text, q)) score += 15;

            const wps = parseFloat(item.wps_score || 0);
            score += wps * 20;

            const matchedCount =
                queryWords.filter(w => title.includes(w) || text.includes(w)).length;

            if (matchedCount === 1 && queryWords.length > 1) {
                score -= 200;
            }

            return score;
        }

        function highlight(text, query) {
            if (!query.trim()) return text;

            let words = query.split(/\s+/).filter(w => w.length > 1);

            words.forEach(word => {
                const short = word.slice(0, Math.ceil(word.length * 0.6));
                const pattern = text.toLowerCase().includes(word.toLowerCase()) ?
                    word :
                    short;

                const reg = new RegExp(`(${pattern})(?!(?:[^<]*?>))`, "gi");
                text = text.replace(
                    reg,
                    '<span class="inline-block bg-yellow-200 text-red-600 px-1 rounded">$1</span>'
                );
            });

            return text;
        }

        function renderSuggestionList(list, query) {
            suggestionBox.classList.remove("hidden");

            if (!list.length) {
                suggestionBox.innerHTML =
                    `<div class="px-4 py-3 text-sm text-gray-500">
                    No results found
                 </div>`;
                isCalculating = false;
                return;
            }

            const sortedList = list.sort((a, b) => b._score - a._score);

            let html = "";
            let lastWasCategory = false;

            sortedList.forEach((item, index) => {
                const isCategory = !!item.term_id;
                const isPlugin = !!item.post_id;

                if (isCategory && !lastWasCategory && index > 0) {
                    html += `<div class="border-t border-gray-200 my-1"></div>`;
                }

                if (isCategory && (index === 0 || !lastWasCategory)) {
                    html += `
                    <div class="px-4 py-2 text-xs font-semibold text-gray-400 uppercase">
                        Category
                    </div>`;
                }

                if (isPlugin && (index === 0 || lastWasCategory)) {
                    if (index > 0) {
                        html += `<div class="border-t border-gray-200 my-1"></div>`;
                    }
                    html += `
                    <div class="px-4 py-2 text-xs font-semibold text-gray-400 uppercase">
                        Plugin
                    </div>`;
                }

                html += `
                <a href="${item.link}"
                   class="block px-4 py-3 hover:bg-gray-100 cursor-pointer text-sm">
                    <span class="font-medium">
                        ${highlight(item.title, query)}
                    </span><br>
                    <span class="text-gray-500 text-xs">
                        ${highlight(item.text, query)}
                    </span>
                </a>
            `;

                lastWasCategory = isCategory;
            });

            suggestionBox.innerHTML = html;
            isCalculating = false;
        }

        function showLoading() {
            suggestionBox.innerHTML = `
            <div class="px-4 py-3 text-sm text-gray-500 flex items-center space-x-2">
                <svg class="animate-spin h-4 w-4 text-gray-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                    <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                </svg>
                <span>Searching...</span>
            </div>`;
            suggestionBox.classList.remove("hidden");
        }

        async function performSearch(query) {
            if (isCalculating) return;

            isCalculating = true;
            showLoading();

            await new Promise(resolve => setTimeout(resolve, 50));

            const rankedResults = globalSuggestions
                .map(item => ({
                    ...item,
                    _score: getPriorityScore(item, query)
                }))
                .filter(item => item._score > 0);

            renderSuggestionList(rankedResults, query);
        }

        searchInput.addEventListener("input", function() {
            const q = searchInput.value.trim();

            if (searchTimer) clearTimeout(searchTimer);

            if (!q) {
                suggestionBox.classList.add("hidden");
                isCalculating = false;
                return;
            }

            if (!suggestionsLoaded || !globalSuggestions.length) {
                suggestionBox.innerHTML =
                    `<div class="px-4 py-3 text-sm text-gray-500">
                    Loading suggestions...
                 </div>`;
                suggestionBox.classList.remove("hidden");
                return;
            }

            showLoading();

            searchTimer = setTimeout(() => {
                performSearch(q);
            }, 150);
        });

        searchInput.addEventListener("focus", () => {
            if (globalSuggestions.length > 0) {
                const allWithScore = globalSuggestions.map(item => ({
                    ...item,
                    _score: 1
                }));
                renderSuggestionList(allWithScore, "");
            }
        });

        document.addEventListener("click", (e) => {
            if (!e.target.closest(".search-wrapper")) {
                suggestionBox.classList.add("hidden");
            }
        });
    }
</script>

<script>
    jQuery(function($) {

        const isCategory = isCategoryPage();

        function triggerFuzzySearch(inputSelector) {
            var searchText = $(inputSelector).val().trim();

            // Save search history
            let history = JSON.parse(localStorage.getItem('search_history') || '[]');

            if (searchText && !history.includes(searchText)) {
                history.push(searchText);
            }

            if (history.length > 20) {
                history = history.slice(-20);
            }

            localStorage.setItem('search_history', JSON.stringify(history));

            if (!isCategory) {
                // NON-category → redirect
                window.location.href =
                    "<?php echo site_url(); ?>/fuzzy-search/?text=" +
                    encodeURIComponent(searchText);
            } else {
                // CATEGORY → submit filter
                $('[name="search_text"]').val(searchText);
                $("#filter_form").submit();
            }
        }

        /* =========================
           CLICK SEARCH BUTTON
        ========================== */

        $('.home_fuzzy_search').on('click', function(e) {
            e.preventDefault();
            const $input = $('#searchInput:visible, #searchInputM:visible').first();
            triggerFuzzySearch($input);
        });

        /* =========================
           ENTER KEY (Desktop + Mobile)
        ========================== */

        $('#searchInput, #searchInputM').on('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                triggerFuzzySearch(this);
            }
        });

    });

    jQuery(function($) {
        $(document).on('click', 'img.close_icon', function() {
            $("#searchInput").val('').trigger('input');
            $("#searchInputM").val('').trigger('input');
        });
    });

    jQuery(document).ready(function($) {

        const $container = $('#search_history_div_show');
        if (!$container.length) return;

        let history = [];

        try {
            history = JSON.parse(localStorage.getItem('search_history')) || [];
        } catch (e) {
            history = [];
        }

        function renderHistory() {
            $container.empty();

            $.each(history, function(index, text) {
                const html = `
                <div class="flex justify-between items-center space-x-3 cursor-pointer">
                    <div class="flex items-center space-x-3 history-search" data-text="${encodeURIComponent(text)}">
                        <img class="w-6 min-w-6 h-6 object-contain object-center"
                             src="<?php echo asset_image_path_generator(); ?>history-line.svg"
                             alt="history-line" />
                        <p>${text}</p>
                    </div>

                    <button class="outline-none remove-history" data-index="${index}">
                        <img class="w-6 min-w-6 h-6 object-contain object-center"
                             src="<?php echo asset_image_path_generator(); ?>close-line.svg"
                             alt="close-line" />
                    </button>
                </div>
            `;
                $container.append(html);
            });
        }

        // Initial render
        renderHistory();

        /* ----------------------------
         * Click history → redirect
         * ---------------------------- */
        $container.on('click', '.history-search', function() {
            const text = $(this).data('text');
            window.location.href =
                "<?php echo site_url(); ?>/fuzzy-search/?text=" + text;
        });

        /* ----------------------------
         * Remove single history item
         * ---------------------------- */
        $container.on('click', '.remove-history', function(e) {
            e.stopPropagation();

            const index = $(this).data('index');

            history.splice(index, 1);
            localStorage.setItem('search_history', JSON.stringify(history));

            renderHistory();
        });

        $(".sortItem").on('click', function(e) {
            $(`[name="sort_by"]`).val($(this).data("short_item"));
            $('#filter_form').submit();
        });

    });
</script>