<?php
class plugin_cpt_frontend_load
{
    /**
     * Enhanced plugin retrieval with flexible category/tag filtering
     * 
     * @param int|array $category_id Primary filter - one or more category IDs (REQUIRED)
     * @param int $limit Maximum number of results
     * @param int $offset Pagination offset
     * @param array $dynamic_filters Additional field filters (rating, downloads, etc.)
     * @param array $order_by Sorting configuration
     * @param string|null $flag_column Flag column to filter by
     * @param mixed $flag_value Value for flag column
     * @param array $primary_order_post_ids Post IDs to prioritize in results
     * @param int|array $post_tag_id OPTIONAL - Narrow results by tag intersection
     * @param string $tag_operation How to combine tags: 'AND' (all tags) or 'OR' (any tag). Default: 'AND'
     * 
     * @return array Filtered plugin results
     */
    public static function get_plugins_with_category_id_filter(
        $category_id,
        $limit = 20,
        $offset = 0,
        $dynamic_filters = [],
        $order_by = [],
        $flag_column = null,
        $flag_value = null,
        $primary_order_post_ids = [],
        $post_tag_id = [],
        $tag_operation = 'AND'
    ) {
        global $wpdb;

        $tbl = $wpdb->prefix . "plugins_core_information";
        $rel_table = $wpdb->prefix . "term_relationships";
        $tax_table = $wpdb->prefix . "term_taxonomy";

        /* ---------------------------------------------------------
         * 0. Extract wp_post_id filter EARLY (hard constraint)
         * --------------------------------------------------------- */
        $post_id_filter = null;

        foreach ($dynamic_filters as $k => $filter) {
            if (($filter['field'] ?? '') === 'wp_post_id' && !empty($filter['value'])) {
                $post_id_filter = array_values(array_unique(array_filter(
                    array_map('intval', (array) $filter['value']),
                    fn($id) => $id > 0
                )));
                unset($dynamic_filters[$k]);
                break;
            }
        }

        /* ---------------------------------------------------------
         * 1. Normalize category IDs (REQUIRED - Primary filter)
         * --------------------------------------------------------- */
        $category_ids = array_values(array_unique(array_filter(
            array_map('intval', (array) $category_id),
            fn($id) => $id > 0
        )));

        if (empty($category_ids)) {
            return []; // No valid categories = no results
        }

        /* ---------------------------------------------------------
         * 2. Resolve post IDs from CATEGORY (Base dataset)
         * --------------------------------------------------------- */
        $cat_sql = "
        SELECT DISTINCT tr.object_id
        FROM {$rel_table} tr
        INNER JOIN {$tax_table} tt
            ON tt.term_taxonomy_id = tr.term_taxonomy_id
        WHERE tt.taxonomy = 'plugin_category'
          AND tt.term_id IN (" . implode(',', $category_ids) . ")
    ";

        $post_ids = array_map('intval', $wpdb->get_col($cat_sql));

        if (empty($post_ids)) {
            return []; // No posts match categories
        }

        /* ---------------------------------------------------------
         * 3. TAG FILTERING (OPTIONAL - Narrows category results)
         * --------------------------------------------------------- */
        $tag_ids = array_values(array_unique(array_filter(
            array_map('intval', (array) $post_tag_id),
            fn($id) => $id > 0
        )));

        if (!empty($tag_ids)) {
            $tag_operation = strtoupper($tag_operation) === 'OR' ? 'OR' : 'AND';

            if ($tag_operation === 'AND') {
                // ALL tags must be present (intersection)
                foreach ($tag_ids as $tag_id) {
                    $tag_sql = "
                    SELECT tr.object_id
                    FROM {$rel_table} tr
                    INNER JOIN {$tax_table} tt
                        ON tt.term_taxonomy_id = tr.term_taxonomy_id
                    WHERE tt.taxonomy = 'plugin_tag'
                      AND tt.term_id = {$tag_id}
                      AND tr.object_id IN (" . implode(',', $post_ids) . ")
                ";

                    $post_ids = array_values(array_intersect(
                        $post_ids,
                        array_map('intval', $wpdb->get_col($tag_sql))
                    ));

                    if (empty($post_ids)) {
                        return []; // No posts match all required tags
                    }
                }
            } else {
                // ANY tag can be present (union)
                $tag_sql = "
                SELECT DISTINCT tr.object_id
                FROM {$rel_table} tr
                INNER JOIN {$tax_table} tt
                    ON tt.term_taxonomy_id = tr.term_taxonomy_id
                WHERE tt.taxonomy = 'plugin_tag'
                  AND tt.term_id IN (" . implode(',', $tag_ids) . ")
                  AND tr.object_id IN (" . implode(',', $post_ids) . ")
            ";

                $tagged_posts = array_map('intval', $wpdb->get_col($tag_sql));
                $post_ids = array_values(array_intersect($post_ids, $tagged_posts));

                if (empty($post_ids)) {
                    return []; // No posts match any tags
                }
            }
        }

        /* ---------------------------------------------------------
         * 4. Apply wp_post_id filter (Additional constraint)
         * --------------------------------------------------------- */
        if (!empty($post_id_filter)) {
            $post_ids = array_values(array_intersect($post_ids, $post_id_filter));
            if (empty($post_ids)) {
                return []; // No overlap with specified post IDs
            }
        }

        /* ---------------------------------------------------------
         * 5. Build WHERE clause with dynamic filters
         * --------------------------------------------------------- */
        $where_parts = [];
        $where_parts[] = "wp_post_id IN (" . implode(',', $post_ids) . ")";

        $allowed_columns = [
            'rating',
            'downloads',
            'updated_at',
            'newest',
            'active_installs',
            'overall_wpscore',
            'performance_impact',
            'plugin_age_days',
            'verified',
            'is_sponsored',
            'is_featured',
            'status',
            'type',
            'support_quality',
            'user_review_sentiment',
            'localization_count',
            'last_update_days',
        ];

        foreach ($dynamic_filters as $filter) {
            if (empty($filter['field']) || empty($filter['operator']))
                continue;
            if (!in_array($filter['field'], $allowed_columns, true))
                continue;

            $field = esc_sql($filter['field']);
            $op = strtolower($filter['operator']);
            $val = $filter['value'];

            if ($op === 'between' && is_array($val) && count($val) === 2) {
                $where_parts[] = "{$field} BETWEEN " . floatval($val[0]) . " AND " . floatval($val[1]);
            } elseif (in_array($op, ['>', '<', '>=', '<='], true)) {
                $where_parts[] = "{$field} {$op} " . floatval($val);
            } elseif (in_array($op, ['=', '!='], true)) {
                $where_parts[] = is_numeric($val)
                    ? "{$field} {$op} " . intval($val)
                    : "{$field} {$op} '" . esc_sql($val) . "'";
            }
        }

        if (!empty($flag_column)) {
            $allowed_flags = ['is_sponsored', 'is_featured', 'status', 'type', 'verified'];
            if (in_array($flag_column, $allowed_flags, true)) {
                $where_parts[] = is_numeric($flag_value)
                    ? "{$flag_column} = " . intval($flag_value)
                    : "{$flag_column} = '" . esc_sql($flag_value) . "'";
            }
        }

        $where = "WHERE " . implode(' AND ', $where_parts);

        /* ---------------------------------------------------------
         * 6. SELECT COLUMNS
         * --------------------------------------------------------- */
        $select_columns = "
        plugin_icon,
        post_title,
        author_name,
        author_profile,
        overall_wpscore,
        rating,
        num_ratings,
        active_installs,
        plugin_age_days,
        tested,
        wp_post_id
    ";

        /* ---------------------------------------------------------
         * 7. NORMALIZE primary_order_post_ids (within filtered set)
         * --------------------------------------------------------- */
        $primary_order_post_ids = array_values(array_unique(array_filter(
            array_map('intval', (array) $primary_order_post_ids),
            fn($id) => $id > 0 && in_array($id, $post_ids, true)
        )));

        /* ---------------------------------------------------------
         * 8. BUILD ORDER BY CLAUSE
         * --------------------------------------------------------- */
        $use_php_sorting = !empty($primary_order_post_ids);
        $orderClause = '';

        if ($use_php_sorting) {
            // When primary IDs exist: prioritize them, then NO additional MySQL sorting
            // (PHP will handle the actual field sorting after retrieval)
            $orderClause = "ORDER BY FIELD(wp_post_id, " . implode(',', $primary_order_post_ids) . ") ASC";
        } else {
            // MySQL sorting when NO primary_order_post_ids
            $order_parts = [];

            foreach ($order_by as $sort) {
                if (empty($sort['field']))
                    continue;
                $dir = strtolower($sort['direction'] ?? 'desc');
                $dir = in_array($dir, ['asc', 'desc'], true) ? $dir : 'desc';
                $order_parts[] = esc_sql($sort['field']) . " {$dir}";
            }

            if (empty($order_parts)) {
                $order_parts[] = "overall_wpscore DESC";
            }

            $orderClause = "ORDER BY " . implode(', ', $order_parts);
        }

        /* ---------------------------------------------------------
         * 9. Execute query
         * --------------------------------------------------------- */
        $sql = "
        SELECT {$select_columns}
        FROM {$tbl}
        {$where}
        {$orderClause}
        LIMIT {$limit} OFFSET {$offset}
    ";

        $results = $wpdb->get_results($sql, ARRAY_A) ?: [];

        /* ---------------------------------------------------------
         * 10. PHP SORTING (when primary_order_post_ids exist)
         * Apply field-based sorting AFTER priority retrieval
         * --------------------------------------------------------- */
        if ($use_php_sorting && !empty($results)) {
            // Get selected sort field (from mapping)
            $sort_field = $order_by[0]['field'] ?? '';
            $sort_dir = strtolower($order_by[0]['direction'] ?? 'desc');

            // If "Relevance" (no field), keep DB order
            if (!empty($sort_field)) {
                usort($results, function ($a, $b) use ($sort_field, $sort_dir) {
                    $aVal = $a[$sort_field] ?? null;
                    $bVal = $b[$sort_field] ?? null;

                    // Handle nulls safely
                    if ($aVal === $bVal) {
                        return 0;
                    }

                    if ($aVal === null) {
                        return 1;
                    }

                    if ($bVal === null) {
                        return -1;
                    }

                    // Numeric comparison
                    if (is_numeric($aVal) && is_numeric($bVal)) {
                        return $sort_dir === 'asc'
                            ? $aVal <=> $bVal
                            : $bVal <=> $aVal;
                    }

                    // String comparison fallback
                    return $sort_dir === 'asc'
                        ? strcmp((string) $aVal, (string) $bVal)
                        : strcmp((string) $bVal, (string) $aVal);
                });
            }
        }

        return $results;
    }

    public static function get_plugins_with_category_id_filter_exclude(
        $category_id,
        $exclude_id,
        $limit,
        $offset,
        $filters = null,
        $flag_column = null,
        $flag_value = null
    ) {
        global $wpdb;
        $tbl = $wpdb->prefix . "plugins_core_information";
        $terms_table = $wpdb->prefix . "term_relationships";
        $post_ids = $wpdb->get_col($wpdb->prepare(
            "SELECT object_id FROM {$terms_table} WHERE term_taxonomy_id = %d",
            $category_id
        ));

        if (empty($post_ids)) {
            // No posts found for this category → return empty
            return [];
        }

        // Convert to SQL-safe IN() list: (1,2,3,4)
        $post_ids_sql = "(" . implode(",", array_map('intval', $post_ids)) . ")";
        // --- Sanitize inputs ---
        $category_id = esc_sql($category_id);
        $exclude_id = intval($exclude_id);
        $limit = intval($limit);
        $offset = intval($offset);

        // --- Build WHERE parts ---
        $where_parts = [];
        $where_parts[] = "wp_post_id IN {$post_ids_sql}";

        // Exclude specific ID (from wp_post_id)
        $where_parts[] = "id != {$exclude_id}";

        // ✅ Flag column filtering (supports numeric/string)
        if (!empty($flag_column)) {
            $allowed_flags = ['is_sponsored', 'is_featured', 'status', 'type', 'verified'];
            if (in_array($flag_column, $allowed_flags)) {
                $flag_column = esc_sql($flag_column);
                if (is_numeric($flag_value)) {
                    $where_parts[] = "{$flag_column} = " . intval($flag_value);
                } elseif (!is_null($flag_value)) {
                    $where_parts[] = "{$flag_column} = '" . esc_sql($flag_value) . "'";
                }
            }
        }

        // Combine WHERE
        $where = "WHERE " . implode(' AND ', $where_parts);

        // --- Build ORDER BY dynamically ---
        $allowed_filters = [
            'rating',
            'downloads',
            'updated_at',
            'newest',
            'active_installs',
            'overall_wpscore',
            'performance_impact',
            'plugin_age_days',
        ];
        $order_parts = [];

        if (!empty($filters)) {
            if (is_string($filters)) {
                $filters = [$filters];
            }

            foreach ($filters as $filter) {
                $parts = explode(':', $filter);
                $column = esc_sql(trim($parts[0]));
                $direction = isset($parts[1]) ? strtolower(trim($parts[1])) : 'desc';

                if (in_array($column, $allowed_filters)) {
                    $direction = in_array($direction, ['asc', 'desc']) ? $direction : 'desc';
                    $order_parts[] = "{$column} {$direction}";
                }
            }
        }

        // ✅ Always add active_installs DESC as fallback
        $order_parts[] = "active_installs DESC";

        $orderClause = "ORDER BY " . implode(', ', $order_parts);

        // --- Select columns ---
        $columns = "
        plugin_icon,
        post_title,
        author_name,
        author_profile,
        overall_wpscore,
        rating,
        num_ratings,
        active_installs,
        tested,
        wp_post_id
    ";

        // --- Build final SQL ---
        $sql = "SELECT {$columns} FROM {$tbl} {$where} {$orderClause}";

        // --- Add LIMIT/OFFSET ---
        if ($limit > 0) {
            $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $limit, $offset);
        }

        // --- Execute ---
        $results = $wpdb->get_results($sql, ARRAY_A);

        return $results ?: [];
    }
    public static function get_the_relevent_comments($slug, $type)
    {
        global $wpdb;
        $tbl_plugin_comments = $wpdb->prefix . "plugin_comments";

        // Prepare a query based on the type
        if ($type == "relevent") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY RAND() LIMIT 3",
                $slug
            );
        }
        if ($type == "newest") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY id ASC LIMIT 3",
                $slug
            );
        }
        if ($type == "highest") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY commenter_num_rating DESC, id ASC LIMIT 3",
                $slug
            );
        }
        if ($type == "lowest") {
            $sql = $wpdb->prepare(
                "SELECT * FROM $tbl_plugin_comments WHERE plugin_slug = %s ORDER BY commenter_num_rating ASC, id ASC LIMIT 3",
                $slug
            );
        }
        //return $sql;
        // Execute the query
        $results = $wpdb->get_results($sql);

        return $results;
    }
    public static function get_the_plugin_comments($slug)
    {
        global $wpdb;
        $tbl = $wpdb->prefix . "plugin_comments";

        // 1) Fetch latest 30 comments sorted by date
        $sql = $wpdb->prepare(
            "SELECT * FROM {$tbl}
         WHERE plugin_slug = %s
         ORDER BY STR_TO_DATE(comment_date, '%%M %%d, %%Y') DESC
         LIMIT 30",
            $slug
        );

        $rows = $wpdb->get_results($sql, ARRAY_A);

        if (empty($rows)) {
            return [
                "relevent" => [],
                "newest" => [],
                "highest" => [],
                "lowest" => [],
            ];
        }

        // Normalize fields
        $comments = [];
        foreach ($rows as $r) {
            $r['commenter_num_rating'] = (int) preg_replace('/\D/', '', $r['commenter_num_rating']);
            $r['id'] = (int) $r['id'];

            // Convert comment_date to timestamp using DateTime for reliability
            try {
                $date = DateTime::createFromFormat('F j, Y', $r['comment_date']);
                $r['comment_timestamp'] = $date ? $date->getTimestamp() : 0;
            } catch (Exception $e) {
                $r['comment_timestamp'] = 0;
            }

            $comments[] = $r;
        }

        // -----------------------------------------------
        // 2) Sort and limit to 3 most recent for each category
        // -----------------------------------------------

        // RELEVANT → 3 random from most recent comments
        $relevent = $comments;
        shuffle($relevent);
        $relevent = array_slice($relevent, 0, 3);

        // NEWEST → top 3 by comment_date (already sorted DESC from query)
        $newest = array_slice($comments, 0, 3);

        // HIGHEST → sort by rating DESC, then by most recent date for tie-breaker
        $highest = $comments;
        usort($highest, function ($a, $b) {
            if ($b['commenter_num_rating'] == $a['commenter_num_rating']) {
                return $b['comment_timestamp'] <=> $a['comment_timestamp']; // most recent first
            }
            return $b['commenter_num_rating'] <=> $a['commenter_num_rating'];
        });
        $highest = array_slice($highest, 0, 3);

        // LOWEST → sort by rating ASC, then by most recent date for tie-breaker
        $lowest = $comments;
        usort($lowest, function ($a, $b) {
            if ($a['commenter_num_rating'] == $b['commenter_num_rating']) {
                return $b['comment_timestamp'] <=> $a['comment_timestamp']; // most recent first
            }
            return $a['commenter_num_rating'] <=> $b['commenter_num_rating'];
        });
        $lowest = array_slice($lowest, 0, 3);

        // -----------------------------------------------
        // 3) RETURN ALL SORTED ARRAYS (MAX 3 EACH)
        // -----------------------------------------------
        return [
            "relevent" => array_values($relevent),
            "newest" => array_values($newest),
            "highest" => array_values($highest),
            "lowest" => array_values($lowest),
        ];
    }
    public static function get_plugins_by_ids(array $ids = [])
    {
        //return  $ids;
        global $wpdb;

        // Skip if empty
        if (empty($ids)) {
            return [];
        }

        // Force ints & clean
        $ids = array_map('intval', $ids);
        $ids = array_filter($ids);

        if (empty($ids)) {
            return [];
        }

        $table = $wpdb->prefix . "plugins_core_information";

        // Same columns you use in your main query
        $columns = "
        plugin_icon,
        post_title,
        author_name,
        author_profile,
        overall_wpscore,
        rating,
        num_ratings,
        active_installs,
        tested,
        post_short_descripton,
        wp_post_id
    ";

        // Build placeholders
        $placeholders = implode(', ', array_fill(0, count($ids), '%d'));

        // SQL
        $sql = $wpdb->prepare(
            "SELECT {$columns}
         FROM {$table}
         WHERE id IN ($placeholders)",
            $ids
        );
        //return $sql;
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }
    public static function fuzzy_search_plugin_suggestions_with_category($search_text)
    {
        if (empty($search_text)) {
            return [];
        }

        /* -------------------------
         * CONFIG
         * ------------------------- */
        $MAX_RESULTS = 100;      // 🔢 limit results
        $MIN_SCORE = 60;       // 🚫 ignore weak matches

        /* -------------------------
         * Load JSON suggestions
         * ------------------------- */
        $file_path = get_stylesheet_directory() . '/plugins-suggestions.json';
        if (!file_exists($file_path)) {
            return [];
        }

        $suggestions = json_decode(file_get_contents($file_path), true);
        if (!is_array($suggestions)) {
            return [];
        }

        /* -------------------------
         * Load TOP (PARENT) categories only
         * ------------------------- */
        $top_categories = get_option('top_plugin_categories');
        if (!is_array($top_categories)) {
            $top_categories = [];
        }

        // ✅ keep ONLY parent categories
        $parent_categories = [];
        foreach ($top_categories as $cat) {
            if (!empty($cat['parent']) && (int) $cat['parent'] !== 0) {
                continue; // skip subcategories
            }

            $parent_categories[] = [
                'id' => (int) $cat['term_id'],
                'name' => mb_strtolower($cat['name'] ?? '', 'UTF-8'),
                'slug' => mb_strtolower($cat['slug'] ?? '', 'UTF-8'),
            ];
        }

        /* -------------------------
         * FUZZY MATCH (unchanged)
         * ------------------------- */
        $fuzzyMatch = function ($str, $query) {
            $str = mb_strtolower($str, 'UTF-8');
            $query = mb_strtolower($query, 'UTF-8');
            $words = array_filter(preg_split('/\s+/', $query));

            foreach ($words as $word) {
                $str_words = preg_split('/\s+/', $str);
                $starts = false;

                foreach ($str_words as $w) {
                    if (strpos($w, $word) === 0) {
                        $starts = true;
                        break;
                    }
                }

                if ($starts)
                    continue;

                if (mb_strlen($word, 'UTF-8') <= 2) {
                    if (strpos($str, $word) === false)
                        return false;
                    continue;
                }

                $partial = mb_substr($word, 0, (int) ceil(mb_strlen($word) * 0.6));
                if (strpos($str, $partial) === false)
                    return false;
            }

            return true;
        };

        /* -------------------------
         * PRIORITY SCORE (unchanged)
         * ------------------------- */
        $getPriorityScore = function ($item, $query) use ($fuzzyMatch) {

            if (trim($query) === '') {
                return 0;
            }

            $q = mb_strtolower(trim($query), 'UTF-8');
            $title = mb_strtolower($item['title'] ?? '', 'UTF-8');
            $text = mb_strtolower($item['text'] ?? '', 'UTF-8');

            $score = 0;

            $queryWords = array_values(array_filter(preg_split('/\s+/', $q)));
            $titleWords = array_values(array_filter(preg_split('/\s+/', $title)));
            $textWords = array_values(array_filter(preg_split('/\s+/', $text)));

            /* =================================================
             * 1️⃣ FULL PHRASE MATCH (VERY HIGH)
             * ================================================= */
            if ($title !== '' && mb_strpos($title, $q) !== false) {
                $score += 400;
            }
            if ($text !== '' && mb_strpos($text, $q) !== false) {
                $score += 200;
            }

            /* =================================================
             * 2️⃣ ALL QUERY WORDS PRESENT (INTENT)
             * ================================================= */
            $allInTitle = true;
            foreach ($queryWords as $w) {
                if (mb_strpos($title, $w) === false) {
                    $allInTitle = false;
                    break;
                }
            }

            $allInText = true;
            foreach ($queryWords as $w) {
                if (mb_strpos($text, $w) === false) {
                    $allInText = false;
                    break;
                }
            }

            if ($allInTitle) {
                $score += 350;
            } elseif ($allInText) {
                $score += 150;
            }

            /* =================================================
             * 3️⃣ WORD PROXIMITY (distance-based)
             * ================================================= */
            if (count($queryWords) > 1 && $title !== '') {
                $joinedTitle = implode(' ', $titleWords);
                $prevPos = null;

                foreach ($queryWords as $w) {
                    $pos = mb_strpos($joinedTitle, $w);
                    if ($pos !== false && $prevPos !== null) {
                        if (abs($pos - $prevPos) < 15) {
                            $score += 150;
                        }
                    }
                    if ($pos !== false) {
                        $prevPos = $pos;
                    }
                }
            }

            /* =================================================
             * 4️⃣ ORDERED WORD MATCH (POSITIONAL)
             * ================================================= */
            $orderedMatch = true;
            foreach ($queryWords as $i => $w) {
                if (!isset($titleWords[$i]) || mb_strpos($titleWords[$i], $w) !== 0) {
                    $orderedMatch = false;
                    break;
                }
            }
            if ($orderedMatch && count($queryWords) > 1) {
                $score += 300;
            }

            /* =================================================
             * 5️⃣ WORD-START MATCH (GENERIC SAFE)
             * ================================================= */
            foreach ($queryWords as $qw) {
                $len = mb_strlen($qw, 'UTF-8');

                foreach ($titleWords as $i => $tw) {
                    if (mb_strpos($tw, $qw) === 0) {
                        $score += ($len > 4) ? 120 : 30;
                        if ($i === 0) {
                            $score += 40;
                        }
                    }
                }

                foreach ($textWords as $tw) {
                    if (mb_strpos($tw, $qw) === 0) {
                        $score += ($len > 4) ? 60 : 15;
                    }
                }
            }

            /* =================================================
             * 6️⃣ FUZZY MATCH (LOW PRIORITY)
             * ================================================= */
            if ($fuzzyMatch($title, $q)) {
                $score += 30;
            }
            if ($fuzzyMatch($text, $q)) {
                $score += 15;
            }

            /* =================================================
             * 7️⃣ SINGLE WORD MATCH PENALTY
             * ================================================= */
            if (count($queryWords) > 1) {
                $matchedWords = 0;
                foreach ($queryWords as $w) {
                    if (mb_strpos($title, $w) !== false || mb_strpos($text, $w) !== false) {
                        $matchedWords++;
                    }
                }

                if ($matchedWords === 1) {
                    $score -= 200;
                }
            }

            return $score;
        };


        /* -------------------------
         * STEP 1: Rank POSTS only
         * ------------------------- */
        $results = [];

        foreach ($suggestions as $item) {

            if (empty($item['post_id']))
                continue;

            $score = $getPriorityScore($item, $search_text);

            // 🚫 Drop weak matches
            if ($score < $MIN_SCORE)
                continue;

            $results[] = [
                'post_id' => (int) $item['post_id'],
                'title' => $item['title'] ?? '',
                'text' => $item['text'] ?? '',
                'score' => $score,
            ];
        }

        /* -------------------------
         * STEP 2: Sort & limit
         * ------------------------- */
        usort($results, fn($a, $b) => $b['score'] <=> $a['score']);
        $results = array_slice($results, 0, $MAX_RESULTS);

        /* -------------------------
         * STEP 3: Attach PARENT category
         * ------------------------- */
        foreach ($results as &$row) {

            $title = mb_strtolower($row['title'], 'UTF-8');
            $text = mb_strtolower($row['text'], 'UTF-8');

            $bestCat = null;
            $bestScore = 0;

            foreach ($parent_categories as $cat) {
                $cs = 0;

                if (strpos($title, $cat['name']) !== false)
                    $cs += 50;
                if (strpos($text, $cat['name']) !== false)
                    $cs += 30;
                if (strpos($title, $cat['slug']) !== false)
                    $cs += 20;

                if ($cs > $bestScore) {
                    $bestScore = $cs;
                    $bestCat = $cat['id'];
                }
            }

            $row['category_id'] = $bestCat;
        }

        /* -------------------------
         * FINAL OUTPUT
         * ------------------------- */
        return array_map(fn($r) => [
            'post_id' => $r['post_id'],
            'category_id' => $r['category_id'],
            'title' => $r['title'],
            'text' => $r['text'],
        ], $results);
    }

    public static function fuzzy_search_plugin_suggestions_with_category_only_post($search_text, $filter_category_id)
    {
        if (empty($search_text) || empty($filter_category_id)) {
            return [];
        }

        /* -------------------------
         * CONFIG
         * ------------------------- */
        $MAX_RESULTS = 100;
        $MIN_SCORE = 60;

        /* -------------------------
         * Load JSON suggestions
         * ------------------------- */
        $file_path = get_stylesheet_directory() . '/plugins-suggestions.json';
        if (!file_exists($file_path)) {
            return [];
        }

        $suggestions = json_decode(file_get_contents($file_path), true);
        if (!is_array($suggestions)) {
            return [];
        }

        /* -------------------------
         * Load TOP (PARENT) categories only
         * ------------------------- */
        $top_categories = get_option('top_plugin_categories');
        if (!is_array($top_categories)) {
            return [];
        }

        $parent_categories = [];
        foreach ($top_categories as $cat) {
            if (!empty($cat['parent']) && (int) $cat['parent'] !== 0)
                continue;

            $parent_categories[] = [
                'id' => (int) $cat['term_id'],
                'name' => mb_strtolower($cat['name'] ?? '', 'UTF-8'),
                'slug' => mb_strtolower($cat['slug'] ?? '', 'UTF-8'),
            ];
        }

        /* -------------------------
         * FUZZY MATCH (JS-equivalent)
         * ------------------------- */
        $fuzzyMatch = function ($str, $query) {
            $str = mb_strtolower($str, 'UTF-8');
            $query = mb_strtolower($query, 'UTF-8');
            $words = array_filter(preg_split('/\s+/', $query));

            foreach ($words as $word) {
                $found = false;
                foreach (preg_split('/\s+/', $str) as $w) {
                    if (mb_strpos($w, $word) === 0) {
                        $found = true;
                        break;
                    }
                }

                if ($found)
                    continue;

                if (mb_strlen($word) <= 2) {
                    if (mb_strpos($str, $word) === false)
                        return false;
                    continue;
                }

                $partial = mb_substr($word, 0, (int) ceil(mb_strlen($word) * 0.6));
                if (mb_strpos($str, $partial) === false)
                    return false;
            }

            return true;
        };

        /* -------------------------
         * PRIORITY SCORE (JS-PARITY)
         * ------------------------- */
        $getPriorityScore = function ($item, $query) use ($fuzzyMatch) {

            $q = mb_strtolower(trim($query), 'UTF-8');
            $title = mb_strtolower($item['title'] ?? '', 'UTF-8');
            $text = mb_strtolower($item['text'] ?? '', 'UTF-8');

            $score = 0;

            $qWords = array_values(array_filter(preg_split('/\s+/', $q)));
            $tWords = array_values(array_filter(preg_split('/\s+/', $title)));
            $xWords = array_values(array_filter(preg_split('/\s+/', $text)));

            /* 1️⃣ FULL PHRASE */
            if ($title !== '' && mb_strpos($title, $q) !== false)
                $score += 400;
            if ($text !== '' && mb_strpos($text, $q) !== false)
                $score += 200;

            /* 2️⃣ ALL WORDS PRESENT */
            $allTitle = true;
            foreach ($qWords as $w) {
                if (mb_strpos($title, $w) === false) {
                    $allTitle = false;
                    break;
                }
            }
            $allText = true;
            foreach ($qWords as $w) {
                if (mb_strpos($text, $w) === false) {
                    $allText = false;
                    break;
                }
            }

            if ($allTitle)
                $score += 350;
            elseif ($allText)
                $score += 150;

            /* 3️⃣ WORD PROXIMITY */
            if (count($qWords) > 1 && $title !== '') {
                $joined = implode(' ', $tWords);
                $prev = null;

                foreach ($qWords as $w) {
                    $pos = mb_strpos($joined, $w);
                    if ($pos !== false && $prev !== null && abs($pos - $prev) < 15) {
                        $score += 150;
                    }
                    if ($pos !== false)
                        $prev = $pos;
                }
            }

            /* 4️⃣ ORDERED WORD MATCH */
            $ordered = true;
            foreach ($qWords as $i => $w) {
                if (!isset($tWords[$i]) || mb_strpos($tWords[$i], $w) !== 0) {
                    $ordered = false;
                    break;
                }
            }
            if ($ordered && count($qWords) > 1)
                $score += 300;

            /* 5️⃣ WORD-START (GENERIC SAFE) */
            foreach ($qWords as $qw) {
                $len = mb_strlen($qw, 'UTF-8');

                foreach ($tWords as $i => $tw) {
                    if (mb_strpos($tw, $qw) === 0) {
                        $score += ($len > 4) ? 120 : 30;
                        if ($i === 0)
                            $score += 40;
                    }
                }

                foreach ($xWords as $xw) {
                    if (mb_strpos($xw, $qw) === 0) {
                        $score += ($len > 4) ? 60 : 15;
                    }
                }
            }

            /* 6️⃣ FUZZY */
            if ($fuzzyMatch($title, $q))
                $score += 30;
            if ($fuzzyMatch($text, $q))
                $score += 15;

            /* 7️⃣ SINGLE WORD PENALTY */
            if (count($qWords) > 1) {
                $matched = 0;
                foreach ($qWords as $w) {
                    if (mb_strpos($title, $w) !== false || mb_strpos($text, $w) !== false) {
                        $matched++;
                    }
                }
                if ($matched === 1)
                    $score -= 200;
            }

            return $score;
        };

        /* -------------------------
         * STEP 1: Rank posts
         * ------------------------- */
        $ranked = [];

        foreach ($suggestions as $item) {
            if (empty($item['post_id']))
                continue;

            $score = $getPriorityScore($item, $search_text);
            if ($score < $MIN_SCORE)
                continue;

            $ranked[] = [
                'post_id' => (int) $item['post_id'],
                'title' => $item['title'] ?? '',
                'text' => $item['text'] ?? '',
                'score' => $score,
            ];
        }

        if (!$ranked)
            return [];

        usort($ranked, fn($a, $b) => $b['score'] <=> $a['score']);
        $ranked = array_slice($ranked, 0, $MAX_RESULTS);

        /* -------------------------
         * STEP 2: Attach & FILTER by parent category
         * ------------------------- */
        $final_ids = [];

        foreach ($ranked as $row) {

            $title = mb_strtolower($row['title'], 'UTF-8');
            $text = mb_strtolower($row['text'], 'UTF-8');

            $bestCat = null;
            $bestScore = 0;

            foreach ($parent_categories as $cat) {
                $cs = 0;
                if (mb_strpos($title, $cat['name']) !== false)
                    $cs += 50;
                if (mb_strpos($text, $cat['name']) !== false)
                    $cs += 30;
                if (mb_strpos($title, $cat['slug']) !== false)
                    $cs += 20;

                if ($cs > $bestScore) {
                    $bestScore = $cs;
                    $bestCat = $cat['id'];
                }
            }

            if ((int) $bestCat === (int) $filter_category_id) {
                $final_ids[] = $row['post_id'];
            }
        }

        return array_values(array_unique($final_ids));
    }

    public static function filter_by_fuzzy_post_id_taxonomy(
        array $post_ids,
        string $taxonomy,
        string $order = 'DESC' // DESC or ASC
    ) {
        global $wpdb;

        // Normalize inputs
        $post_ids = array_values(array_unique(array_filter(
            array_map('intval', $post_ids),
            fn($id) => $id > 0
        )));

        if (empty($post_ids) || empty($taxonomy)) {
            return [];
        }

        $order = strtoupper($order) === 'ASC' ? 'ASC' : 'DESC';
        $post_ids_sql = implode(',', $post_ids);

        $sql = "
        SELECT 
            t.term_id AS id,
            t.name    AS name,
            COUNT(tr.object_id) AS count
        FROM {$wpdb->prefix}term_relationships tr
        INNER JOIN {$wpdb->prefix}term_taxonomy tt
            ON tt.term_taxonomy_id = tr.term_taxonomy_id
        INNER JOIN {$wpdb->prefix}terms t
            ON t.term_id = tt.term_id
        WHERE tr.object_id IN ({$post_ids_sql})
          AND tt.taxonomy = %s
        GROUP BY t.term_id
        ORDER BY count {$order}
    ";

        return $wpdb->get_results(
            $wpdb->prepare($sql, $taxonomy),
            ARRAY_A
        ) ?: [];
    }
    public static function get_all_tags_by_category_ids($category_ids_array)
    {
        // Validate input
        $category_ids = array_values(array_unique(array_filter(
            array_map('intval', (array) $category_ids_array),
            fn($id) => $id > 0
        )));

        if (empty($category_ids)) {
            return [];
        }

        $merged_tags = [];
        $tag_index = []; // Track unique tags by ID
        $position = 0;   // Track order priority

        // Process each category in the order provided
        foreach ($category_ids as $cat_id) {

            // Get tag data from category meta
            $tag_data_json = get_term_meta($cat_id, '_plugin_category_tag_data', true);

            if (empty($tag_data_json)) {
                continue;
            }

            // Decode JSON
            $tag_data = json_decode($tag_data_json, true);

            if (!is_array($tag_data) || empty($tag_data)) {
                continue;
            }

            // Process each tag from this category
            foreach ($tag_data as $tag) {
                $tag_id = intval($tag['id'] ?? 0);

                if ($tag_id <= 0) {
                    continue;
                }

                // If tag already exists, accumulate count and keep first occurrence position
                if (isset($tag_index[$tag_id])) {
                    $existing_index = $tag_index[$tag_id];
                    $merged_tags[$existing_index]['count'] += intval($tag['count'] ?? 0);

                    // Track which categories this tag appears in
                    if (!isset($merged_tags[$existing_index]['categories'])) {
                        $merged_tags[$existing_index]['categories'] = [];
                    }
                    $merged_tags[$existing_index]['categories'][] = $cat_id;

                } else {
                    // New tag - add to merged list
                    $tag_index[$tag_id] = $position;

                    $merged_tags[$position] = [
                        'id' => $tag_id,
                        'name' => $tag['name'] ?? '',
                        'count' => intval($tag['count'] ?? 0),
                        'categories' => [$cat_id], // Track source categories
                        'first_seen_in_category' => $cat_id // Track which category introduced this tag
                    ];

                    $position++;
                }
            }
        }

        // Re-index array to remove gaps (if any)
        return array_values($merged_tags);
    }
}