<?php

class PluginCategoryTaxonomyManagement
{
    /**
     * 1️⃣ Add form — new category
     */
    public static function plugin_category_add_form_fields($taxonomy)
    {
        ?>
        <div class="form-field term-group">
            <label for="plugin_category_image"><?php _e('Featured Image', 'textdomain'); ?></label>
            <input type="hidden" id="plugin_category_image" name="plugin_category_image" value="">
            <div id="plugin_category_image_preview"></div>
            <button type="button"
                class="upload_image_button button"><?php _e('Upload / Select Image', 'textdomain'); ?></button>
        </div>
        <script>
            jQuery(document).ready(function ($) {
                var frame;
                $('.upload_image_button').on('click', function (e) {
                    e.preventDefault();

                    if (frame) {
                        frame.open();
                        return;
                    }

                    frame = wp.media({
                        title: 'Select or Upload Image',
                        button: {
                            text: 'Use this image'
                        },
                        multiple: false
                    });

                    frame.on('select', function () {
                        var attachment = frame.state().get('selection').first().toJSON();
                        $('#plugin_category_image').val(attachment.url); // ✅ store URL, not ID
                        $('#plugin_category_image_preview').html('<img src="' + attachment.url + '" style="max-width:150px;height:auto;">');
                    });

                    frame.open();
                });
            });
        </script>
        <?php
    }

    /**
     * 2️⃣ Edit form — existing category
     */
    public static function plugin_category_edit_form_fields($term)
    {
        $image_url = get_term_meta($term->term_id, 'plugin_category_image', true);
        ?>
        <tr class="form-field term-group-wrap">
            <th scope="row"><label for="plugin_category_image"><?php _e('Featured Image', 'textdomain'); ?></label></th>
            <td>
                <input type="hidden" id="plugin_category_image" name="plugin_category_image"
                    value="<?php echo esc_attr($image_url); ?>">
                <div id="plugin_category_image_preview">
                    <?php if ($image_url): ?>
                        <img src="<?php echo esc_url($image_url); ?>" style="max-width:150px;height:auto;">
                    <?php endif; ?>
                </div>
                <button type="button"
                    class="upload_image_button button"><?php _e('Upload / Select Image', 'textdomain'); ?></button>
            </td>
        </tr>
        <script>
            jQuery(document).ready(function ($) {
                var frame;
                $('.upload_image_button').on('click', function (e) {
                    e.preventDefault();

                    if (frame) {
                        frame.open();
                        return;
                    }

                    frame = wp.media({
                        title: 'Select or Upload Image',
                        button: {
                            text: 'Use this image'
                        },
                        multiple: false
                    });

                    frame.on('select', function () {
                        var attachment = frame.state().get('selection').first().toJSON();
                        $('#plugin_category_image').val(attachment.url); // ✅ store URL
                        $('#plugin_category_image_preview').html('<img src="' + attachment.url + '" style="max-width:150px;height:auto;">');
                    });

                    frame.open();
                });
            });
        </script>
        <?php
    }

    /**
     * 3️⃣ Save the raw image URL
     */
    public static function plugin_category_save_image($term_id)
    {
        if (isset($_POST['plugin_category_image'])) {
            $image_url = esc_url_raw($_POST['plugin_category_image']);
            update_term_meta($term_id, 'plugin_category_image', $image_url);
        }
    }

    /**
     * 4️⃣ Add column right after Name
     */
    public static function plugin_category_add_image_column($columns)
    {
        $new_columns = [];
        foreach ($columns as $key => $value) {
            $new_columns[$key] = $value;
            if ($key === 'name') {
                $new_columns['plugin_category_image'] = __('Image', 'textdomain');
            }
        }
        return $new_columns;
    }

    /**
     * 5️⃣ Render image in column
     */
    public static function plugin_category_image_column_content($content, $column_name, $term_id)
    {
        if ($column_name === 'plugin_category_image') {
            $image_url = get_term_meta($term_id, 'plugin_category_image', true);
            if ($image_url) {
                $content = '<img src="' . esc_url($image_url) . '" style="width:40px;height:auto;border-radius:4px;border:1px solid #ddd;padding:2px;">';
            } else {
                $content = '<span style="color:#999;">No Image</span>';
            }
        }
        return $content;
    }

    /**
     * 6️⃣ Generate missing image JSON files
     */
    public static function generate_missing_image_jsons()
    {
        $taxonomy = 'plugin_category';
        $terms = get_terms([
            'taxonomy' => $taxonomy,
            'parent' => 0,
            'hide_empty' => false,
        ]);

        if (empty($terms) || is_wp_error($terms)) {
            return;
        }

        $missing = [];
        foreach ($terms as $term) {
            $image_url = get_term_meta($term->term_id, 'plugin_category_image', true);
            if (empty($image_url)) {
                $missing[] = [
                    'term_id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                    'image_url' => '', // to be filled later
                ];
            }
        }

        if (empty($missing))
            return;

        $theme_dir = get_stylesheet_directory();
        $json_dir = $theme_dir . '/plugin-category-image-jsons';
        if (!file_exists($json_dir)) {
            wp_mkdir_p($json_dir);
        }

        $chunks = array_chunk($missing, 25);
        foreach ($chunks as $index => $chunk) {
            $file = $json_dir . '/missing-images-page-' . ($index + 1) . '.json';
            file_put_contents($file, json_encode($chunk, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
        }
    }

    /**
     * 7️⃣ Admin menu under Tools
     */
    public static function add_json_manager_menu()
    {
        add_management_page(
            'Plugin Category Images',
            'Plugin Category Images',
            'manage_options',
            'plugin-category-images',
            ['PluginCategoryTaxonomyManagement', 'render_json_files_page'],
            'dashicons-images-alt2',
            81
        );
    }

    /**
     * 8️⃣ Manage JSON file editing
     */
    public static function render_json_files_page()
    {
        $dir = get_stylesheet_directory() . '/plugin-category-image-jsons/';
        $files = glob($dir . '*.json');
        $selected_file = isset($_GET['file']) ? sanitize_text_field($_GET['file']) : '';
        $file_path = $dir . $selected_file;

        echo '<div class="wrap"><h1>Plugin Category Missing Images</h1><hr/>';

        if (empty($selected_file)) {
            echo '<h2>Available Files</h2><ul>';
            if (!empty($files)) {
                foreach ($files as $file) {
                    $basename = basename($file);
                    $edit_url = admin_url('admin.php?page=plugin-category-images&file=' . urlencode($basename));
                    echo '<li><a href="' . esc_url($edit_url) . '">' . esc_html($basename) . '</a></li>';
                }
            } else {
                echo '<li><em>No JSON files found.</em></li>';
            }
            echo '</ul></div>';
            return;
        }

        if (!file_exists($file_path)) {
            echo '<p style="color:red;">File not found.</p></div>';
            return;
        }

        if (isset($_POST['save_json_file']) && check_admin_referer('save_json_file_action', 'save_json_file_nonce')) {
            $new_content = wp_unslash($_POST['json_content']);
            file_put_contents($file_path, $new_content);
            echo '<div class="updated notice"><p><strong>File saved successfully!</strong></p></div>';
        }

        $content = file_get_contents($file_path);
        echo '<h2>Editing: ' . esc_html($selected_file) . '</h2>';
        echo '<form method="post">';
        wp_nonce_field('save_json_file_action', 'save_json_file_nonce');
        echo '<textarea name="json_content" rows="30" style="width:100%;font-family:monospace;">' . esc_textarea($content) . '</textarea>';
        echo '<p><input type="submit" name="save_json_file" class="button button-primary" value="Save File"></p>';
        echo '</form>';
        echo '<hr/><p><a href="' . esc_url(admin_url('admin.php?page=plugin-category-images')) . '">← Back to File List</a></p></div>';
    }

    /**
     * 9️⃣ Process JSON imports (no upload)
     */
    public static function process_image_import_from_json($file_path)
    {
        if (!file_exists($file_path))
            return false;

        $content = file_get_contents($file_path);
        $data = json_decode($content, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data))
            return false;

        foreach ($data as $entry) {
            if (empty($entry['term_id']) || empty($entry['image_url']))
                continue;

            update_term_meta(
                intval($entry['term_id']),
                'plugin_category_image',
                esc_url_raw($entry['image_url'])
            );
        }

        // After processing, delete the file immediately
        unlink($file_path);

        return true;
    }

    public static function process_all_json_files()
    {
        $dir = get_stylesheet_directory() . '/plugin-category-image-jsons/';

        // Ensure directory exists
        if (!file_exists($dir)) {
            error_log('❌ JSON directory not found: ' . $dir);
            return false;
        }

        // Get all .json files
        $files = glob($dir . '*.json');
        if (empty($files)) {
            error_log('ℹ️ No JSON files found in directory: ' . $dir);
            return false;
        }

        foreach ($files as $file_path) {
            // Process each JSON file
            $result = self::process_image_import_from_json($file_path);

            // Logging
            if ($result) {
                error_log('✅ Processed and deleted file: ' . basename($file_path));
            } else {
                error_log('⚠️ Failed to process file: ' . basename($file_path));
            }
        }

        return true;
    }
    public static function saving_plugin_tag_in_category_meta()
    {
        // Get all parent categories
        $categories = get_terms([
            'taxonomy' => 'plugin_category',
            'hide_empty' => false,
            'parent' => 0,
        ]);

        if (empty($categories) || is_wp_error($categories)) {
            return;
        }

        foreach ($categories as $cat) {

            $cat_id = intval($cat->term_id);

            // 1️⃣ Get all posts assigned to this category
            $posts = get_posts([
                'post_type' => 'plugins', // <-- Change to your CPT name
                'posts_per_page' => -1,
                'tax_query' => [
                    [
                        'taxonomy' => 'plugin_category',
                        'field' => 'term_id',
                        'terms' => $cat_id,
                    ]
                ],
                'fields' => 'ids',
            ]);

            // 2️⃣ No posts in this category
            if (empty($posts)) {
                update_term_meta($cat_id, '_plugin_category_tag_data', wp_json_encode([]));
                update_term_meta($cat_id, '_plugin_category_tag_ids', '');
                continue;
            }

            // 3️⃣ Collect tag IDs for all posts
            $tag_counts = [];

            foreach ($posts as $post_id) {
                $tags = wp_get_post_terms($post_id, 'plugin_tag', ['fields' => 'ids']);

                if (!empty($tags) && !is_wp_error($tags)) {
                    foreach ($tags as $tag_id) {
                        if (!isset($tag_counts[$tag_id])) {
                            $tag_counts[$tag_id] = 0;
                        }
                        $tag_counts[$tag_id]++;
                    }
                }
            }

            // 4️⃣ Build tag data list
            $tag_data = [];

            foreach ($tag_counts as $tag_id => $count) {
                $tag = get_term($tag_id, 'plugin_tag');

                if ($tag && !is_wp_error($tag)) {
                    $tag_data[] = [
                        'id' => $tag_id,
                        'name' => $tag->name,
                        'count' => $count,
                    ];
                }
            }

            // 5️⃣ Sort by usage DESC
            usort($tag_data, fn($a, $b) => $b['count'] <=> $a['count']);

            // 6️⃣ Extract IDs list
            $tag_ids = array_column($tag_data, 'id');
            $csv_ids = implode(',', $tag_ids);

            // 7️⃣ Save to category meta
            update_term_meta($cat_id, '_plugin_category_tag_ids', $csv_ids);
            update_term_meta($cat_id, '_plugin_category_tag_data', wp_json_encode($tag_data));
        }
    }
    public static function saving_plugin_tag_in_category_meta_child_category()
    {
        // Get all parent categories
        $categories = get_terms([
            'taxonomy' => 'plugin_category',
            'hide_empty' => false,
            'parent__not_in' => [0],
        ]);

        if (empty($categories) || is_wp_error($categories)) {
            return;
        }

        foreach ($categories as $cat) {

            $cat_id = intval($cat->term_id);

            // 1️⃣ Get all posts assigned to this category
            $posts = get_posts([
                'post_type' => 'plugins', // <-- Change to your CPT name
                'posts_per_page' => -1,
                'tax_query' => [
                    [
                        'taxonomy' => 'plugin_category',
                        'field' => 'term_id',
                        'terms' => $cat_id,
                    ]
                ],
                'fields' => 'ids',
            ]);

            // 2️⃣ No posts in this category
            if (empty($posts)) {
                update_term_meta($cat_id, '_plugin_category_tag_data', wp_json_encode([]));
                update_term_meta($cat_id, '_plugin_category_tag_ids', '');
                continue;
            }

            // 3️⃣ Collect tag IDs for all posts
            $tag_counts = [];

            foreach ($posts as $post_id) {
                $tags = wp_get_post_terms($post_id, 'plugin_tag', ['fields' => 'ids']);

                if (!empty($tags) && !is_wp_error($tags)) {
                    foreach ($tags as $tag_id) {
                        if (!isset($tag_counts[$tag_id])) {
                            $tag_counts[$tag_id] = 0;
                        }
                        $tag_counts[$tag_id]++;
                    }
                }
            }

            // 4️⃣ Build tag data list
            $tag_data = [];

            foreach ($tag_counts as $tag_id => $count) {
                $tag = get_term($tag_id, 'plugin_tag');

                if ($tag && !is_wp_error($tag)) {
                    $tag_data[] = [
                        'id' => $tag_id,
                        'name' => $tag->name,
                        'count' => $count,
                    ];
                }
            }

            // 5️⃣ Sort by usage DESC
            usort($tag_data, fn($a, $b) => $b['count'] <=> $a['count']);

            // 6️⃣ Extract IDs list
            $tag_ids = array_column($tag_data, 'id');
            $csv_ids = implode(',', $tag_ids);

            // 7️⃣ Save to category meta
            update_term_meta($cat_id, '_plugin_category_tag_ids', $csv_ids);
            update_term_meta($cat_id, '_plugin_category_tag_data', wp_json_encode($tag_data));
        }
    }


}
