<?php
require_once dirname(__FILE__) . '/index.php';

class WpGenerateComparisonPlugins
{

    public function __construct()
    {
        if (!new_post_publishing) {
            echo "No plan to Generate New post so no new AI call";
            update_option('wp_comparison_plugin_create_job_last_run', current_time('mysql'), false);
            return; // already running
        }
        // Check & acquire lock
        if (!wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->run();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — generates comparison posts
     */
    public function run()
    {
        global $wpdb;

        $tbl_plugin_comparisons = $wpdb->prefix . 'plugin_comparisons';

        update_option('wp_comparison_plugin_create_job_last_run', current_time('mysql'), false);

        $plugin_comparisons = $wpdb->get_results("
            SELECT *
            FROM {$tbl_plugin_comparisons}
            WHERE is_ai_comparison_json_done = 1
            AND is_post_created = 0
            LIMIT 100
        ");

        if (empty($plugin_comparisons)) {
            return;
        }

        foreach ($plugin_comparisons as $row) {

            // --------------------------------------------------
            // Parse plugin slugs
            // --------------------------------------------------
            $slugs = array_filter(array_map('trim', explode(',', $row->plugin_slugs)));

            $plugins = $this->get_plugins_name_icon_by_slugs($slugs);

            // --------------------------------------------------
            // Build plugin header + collect icons
            // --------------------------------------------------
            $plugin_icons = [];
            $plugin_header_html = '<div class="comparison-plugins">';

            foreach ($plugins as $plugin) {

                if (!empty($plugin['plugin_icon'])) {
                    $plugin_icons[] = esc_url_raw($plugin['plugin_icon']);
                }

                $plugin_header_html .= '
                    <div class="plugin-item">
                        <img src="' . esc_url($plugin['plugin_icon']) . '" alt="' . esc_attr($plugin['name']) . '" />
                        <h3>' . esc_html($plugin['name']) . '</h3>
                    </div>
                ';
            }

            $plugin_header_html .= '</div>';

            // --------------------------------------------------
            // Final post content
            // --------------------------------------------------
            $post_content = $plugin_header_html;

            // --------------------------------------------------
            // Create post
            // --------------------------------------------------
            $post_data = [
                'post_title' => $row->comparison_title,
                'post_content' => $post_content,
                'post_status' => 'publish',
                'post_type' => 'plugin_comparison',

                'meta_input' => [
                    '_plugin_comparison_data' => $row->plugin_comparison,
                    '_comparison_plugin_slugs' => $row->plugin_slugs,
                    '_comparison_plugin_icons' => implode(',', $plugin_icons),
                ],
            ];

            $post_id = wp_insert_post($post_data, true);

            if (is_wp_error($post_id)) {
                continue;
            }

            // --------------------------------------------------
            // Handle taxonomy mapping
            // --------------------------------------------------
            if (!empty($row->plugin_category_id)) {

                $comparison_term_id = $this->get_or_create_comparison_category(
                    (int) $row->plugin_category_id
                );

                if ($comparison_term_id) {
                    wp_set_object_terms(
                        $post_id,
                        [$comparison_term_id],
                        'comparison_category'
                    );
                }
            }

            // --------------------------------------------------
            // Mark completed
            // --------------------------------------------------
            $wpdb->update(
                $tbl_plugin_comparisons,
                [
                    'is_post_created' => 1,
                    'wp_post_id' => $post_id,
                ],
                ['id' => $row->id],
                ['%d', '%d'],
                ['%d']
            );
        }

        update_option('wp_comparison_plugin_create_job_last_run', current_time('mysql'), false);
    }

    /**
     * Get plugin name & icon by slugs
     */
    private function get_plugins_name_icon_by_slugs(array $slugs)
    {
        global $wpdb;

        if (empty($slugs)) {
            return [];
        }

        $table = $wpdb->prefix . 'plugins_core_information';
        $placeholders = implode(',', array_fill(0, count($slugs), '%s'));

        $query = $wpdb->prepare(
            "SELECT slug, name, plugin_icon
             FROM {$table}
             WHERE slug IN ($placeholders)",
            $slugs
        );

        return $wpdb->get_results($query, ARRAY_A);
    }

    /**
     * Create / get comparison category from plugin_category
     */
    private function get_or_create_comparison_category($plugin_category_id)
    {
        $source_term = get_term($plugin_category_id, 'plugin_category');

        if (!$source_term || is_wp_error($source_term)) {
            return false;
        }

        $new_slug = $source_term->slug . '-comp';

        $existing = get_term_by('slug', $new_slug, 'comparison_category');

        if ($existing && !is_wp_error($existing)) {
            return (int) $existing->term_id;
        }

        $created = wp_insert_term(
            $source_term->name,
            'comparison_category',
            [
                'slug' => $new_slug,
            ]
        );

        if (is_wp_error($created)) {
            return false;
        }

        return (int) $created['term_id'];
    }
}

// --------------------------------------------------
// Run
// --------------------------------------------------
new WpGenerateComparisonPlugins();
