<?php
require_once dirname(__FILE__) . '/index.php';

class WpPluginAIProfile
{

    public function __construct()
    {
        //$this->run();
        //Check & acquire lock
        if (! wp_job_lock()) {
            echo "already running";
            return; // already running
        }

        try {
            // Call core function ONLY after lock
            $this->run();
        } finally {
            // Always release lock
            wp_job_unlock();
        }
    }

    /**
     * Main process — updates active_installs in cron_scraped_plugins_log
     */
    public function run()
    {
        global $wpdb;

        $tbl_plugins_core_information = $wpdb->prefix . "plugins_core_information";

        // ✅ Job start time
        update_option('wp_plugin_ai_profile_job_last_run', current_time('mysql'), false);

        $plugins_core_information = $wpdb->get_results("
            SELECT
                id, slug, name
            FROM {$tbl_plugins_core_information}
            WHERE
                is_wp_post_created = 1
                AND plugin_ai_profile IS NULL
            LIMIT 50
        ");

        if (empty($plugins_core_information)) {
            echo "No Plugins found.\n";
            return;
        }

        foreach ($plugins_core_information as $single_plugin_information) {

            $get_open_ai_ai_profile = $this->get_open_ai_ai_profile(
                $single_plugin_information->id,
                $single_plugin_information->name,
                $single_plugin_information->slug
            );
            // dd($get_open_ai_ai_profile );
            // exit;
            if (! $get_open_ai_ai_profile) {
                continue;
            }

            $json_parsed_data = json_decode($get_open_ai_ai_profile, true);
          
            if (! $json_parsed_data) {
                continue;
            }

            $wpdb->update(
                $tbl_plugins_core_information,
                [
                    'plugin_ai_profile' => $get_open_ai_ai_profile,
                ],
                ['id' => $single_plugin_information->id],
                [
                    '%s', // plugin_ai_profile
                ],
                ['%d']
            );
        }

        // ✅ Job end time
        update_option('wp_plugin_ai_profile_job_last_run', current_time('mysql'), false);
    }
    public function get_ai_profile_output_structure()
    {
        $data = [
            "slug"                               => "",
            "name"                               => "",
            "key_features"                       => [
                "Feature one",
                "Feature two",
                "Feature three",
                "Feature four",
                "Feature five",
            ],
            "monthly_traffic"                    => [
                "0-1000"         => "",
                "1000-10000"     => "",
                "10000-50000"    => "",
                "50000-50000000" => "",
            ],
            "comfortable_configuration_settings" => [
                "beginner"     => "",
                "intermediate" => "",
                "expert"       => "",
            ],
        ];

        return json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    }

    public function get_open_ai_ai_profile($id, $name, $slug)
    {
        try {
            $OPENAI_API_KEY = YOUR_OPENAI_API_KEY;
            $API_URL        = "https://api.openai.com/v1/chat/completions";

            // =====================================================
            // PROMPT (MATCHES YOUR EXISTING STYLE)
            // =====================================================
            $prompt = <<<EOT
            You are a WordPress plugin performance and usability analyst.

            Your task is to analyze the plugin below and generate an AI profile.

            --------------------------------------------------
            ### OUTPUT STRUCTURE (MUST MATCH EXACTLY)
            --------------------------------------------------
            {$this->get_ai_profile_output_structure()}

            --------------------------------------------------
            ### STRICT RULES (NON-NEGOTIABLE)
            --------------------------------------------------
            - Respond ONLY with valid JSON
            - Do NOT include explanations, comments, markdown, or extra text
            - Do NOT add, remove, or rename fields
            - Use the provided plugin name and slug EXACTLY
            - key_features MUST contain between 5 and 10 items
            - Each key feature must be 2–4 words
            - Enum values MUST be exactly one of: "High", "Moderate", "Low"
            - NEVER combine enum values
            - If uncertain, prefer "Moderate"

            --------------------------------------------------
            ### KEY CONCEPT DEFINITIONS
            --------------------------------------------------

            monthly_traffic (PERFORMANCE SUITABILITY):
            - This does NOT represent plugin popularity, install count, or brand recognition
            - This represents how suitable the plugin is for a site receiving that level of monthly traffic
            - Assume the plugin runs on typical shared or VPS hosting
            - Consider frontend rendering cost, database queries, asset weight, and runtime overhead
            - Heavy frontend plugins increase performance risk at scale
            - High traffic ranges require conservative evaluation

            IMPORTANT CONSTRAINTS FOR monthly_traffic:
            - Values should generally stay the same or DECREASE as traffic increases
            - Do NOT mark high traffic ranges as "High" unless the plugin is clearly optimized for scale
            - Page builders, visual editors, ecommerce, and dynamic content plugins should be conservative at high traffic ranges
            - If unsure, do NOT assign "High" to large traffic ranges

            ENUM INTERPRETATION:
            - High = well suited or optimized for that traffic range
            - Moderate = usable with reasonable optimization
            - Low = performance risk or not designed for that scale

            --------------------------------------------------
            comfortable_configuration_settings:
            - beginner: minimal setup, safe defaults, non-technical users
            - intermediate: requires familiarity with WordPress settings
            - expert: requires technical knowledge, advanced configuration, or developer usage

            --------------------------------------------------
            ### PLUGIN INPUT DATA
            --------------------------------------------------
            Plugin Name: "{$name}"
            Plugin Slug: "{$slug}"

            --------------------------------------------------
            ### FINAL INSTRUCTION
            --------------------------------------------------
            Return ONLY the JSON object. No explanation. No extra text.

            EOT;

            // -----------------------------------
            // API PAYLOAD (same as your old code)
            // -----------------------------------
            $payload = [
                "model"           => "gpt-4o",
                "messages"        => [
                    ["role" => "user", "content" => $prompt],
                ],
                "temperature"     => 0.4,
                "response_format" => ["type" => "text"],
            ];

            // -----------------------------------
            // CURL REQUEST
            // -----------------------------------
            $ch = curl_init($API_URL);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST           => true,
                CURLOPT_HTTPHEADER     => [
                    "Content-Type: application/json",
                    "Authorization: Bearer {$OPENAI_API_KEY}",
                ],
                CURLOPT_POSTFIELDS => json_encode($payload),
            ]);

            $response = curl_exec($ch);

            if (curl_errno($ch)) {
                throw new Exception('cURL Error: ' . curl_error($ch));
            }

            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($http_code < 200 || $http_code >= 300) {
                throw new Exception("API request failed: {$response}");
            }

            // -----------------------------------
            // PARSE RESPONSE (same regex logic)
            // -----------------------------------
            $data    = json_decode($response, true);
            $content = $data['choices'][0]['message']['content'] ?? '';

            if (preg_match('/\{.*\}/s', $content, $match)) {
                $content = $match[0];
            }

            $json_output = json_decode($content, true);

            if (! $json_output) {
                throw new Exception('Failed to parse AI profile JSON.');
            }

            return json_encode($json_output, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

        } catch (Exception $e) {
            error_log("AI Profile Error: " . $e->getMessage());

            if (isset($response)) {
                $this->enc_save_error($id, $response);
            }

            return false;
        }
    }

    public function enc_save_error($id, $error_content)
    {
        global $wpdb;
        $plugins_core_information_log = $wpdb->prefix . 'plugins_core_information';

        $info = [
            'error_log_open_ai' => $error_content,
        ];
        $where_condition = [
            'id' => $id,
        ];
        $wpdb->update($plugins_core_information_log, $info, $where_condition);
    }

}

// Instantiate the class
new WpPluginAIProfile();
