<?php
/**
 * Settings class
 * Server Scheduler settings
 *
 * @package advanced-cron-manager
 */

namespace underDEV\AdvancedCronManager\Server;

use underDEV\Utils;

/**
 * Settings class.
 */
class Settings {

	/**
	 * View class
	 *
	 * @var object
	 */
	public $view;

	/**
	 * Ajax class
	 *
	 * @var object
	 */
	private $ajax;

	/**
	 * Option name
	 *
	 * @var string
	 */
	private $option_name;

	/**
	 * Defaults
	 *
	 * @var array|int[]
	 */
	private $default;

	/**
	 * Settings
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * Constructor
	 *
	 * @param Utils\View $view View class.
	 * @param Utils\Ajax $ajax Ajax class.
	 */
	public function __construct( Utils\View $view, Utils\Ajax $ajax ) {
		$this->view = $view;
		$this->ajax = $ajax;

		$this->option_name = 'acm_server_settings';

		$this->default = array(
			'server_enable' => 0,
		);
	}

	/**
	 * Loads Server Scheduler settings part
	 *
	 * @return void
	 */
	public function load_settings_part() {
		$this->view->set_var( 'settings', $this->get_settings() );
		$this->view->set_var( 'disable_wp_cron_defined', $this->is_disable_wp_cron_defined() );
		$this->view->get_view( 'server/settings' );
	}

	/**
	 * Gets Settings
	 * Supports lazy loading
	 *
	 * @param  boolean $force if refresh stored events.
	 * @return array          saved settings
	 */
	public function get_settings( $force = false ) {

		if ( empty( $this->settings ) || $force ) {
			$this->settings = get_option( $this->option_name, $this->default );
		}

		return $this->settings;
	}

	/**
	 * Checks if DISABLE_WP_CRON constant is defined
	 *
	 * @return boolean
	 */
	public function is_disable_wp_cron_defined() {
		return defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON;
	}

	/**
	 * Saves settings
	 * Called by AJAX
	 *
	 * @return void
	 */
	public function save_settings() {

		$this->ajax->verify_nonce( 'acm/server/settings/save' );

		if ( ! current_user_can( 'manage_options' ) ) {
			$this->ajax->response( false, array(
				__( "You're not allowed to do that.", 'advanced-cron-manager' ),
			) );
		}

		$errors = array();

		$form_options = array_map( function () {
			return 0;
		}, $this->default );

		// phpcs:ignore
		$form_data = wp_parse_args( $_REQUEST['data'], $form_options );

		// Validate and sanitize settings.
		$sanitized_data = array();
		foreach ( $form_data as $key => $value ) {
			if ( ! array_key_exists( $key, $this->default ) ) {
				continue; // Skip unknown settings.
			}

			// All current settings are boolean (0 or 1).
			$sanitized_data[ $key ] = absint( $value ) === 1 ? 1 : 0;
		}

		update_option( $this->option_name, $sanitized_data );

		$this->ajax->response( __( 'Settings has been saved', 'advanced-cron-manager' ), $errors );
	}
}
