<?php

/**
 * WooCommerce Compatibility File
 *
 * @link https://woocommerce.com/
 *
 * @package tusenrosor-wp-theme
 */

/**
 * WooCommerce setup function.
 *
 * @link https://docs.woocommerce.com/document/third-party-custom-theme-compatibility/
 * @link https://github.com/woocommerce/woocommerce/wiki/Enabling-product-gallery-features-(zoom,-swipe,-lightbox)
 * @link https://github.com/woocommerce/woocommerce/wiki/Declaring-WooCommerce-support-in-themes
 *
 * @return void
 */
function tusenrosor_encoderit_woocommerce_setup()
{
	add_theme_support(
		'woocommerce',
		array(
			'thumbnail_image_width' => 150,
			'single_image_width'    => 300,
			'product_grid'          => array(
				'default_rows'    => 3,
				'min_rows'        => 1,
				'default_columns' => 4,
				'min_columns'     => 1,
				'max_columns'     => 6,
			),
		)
	);
	add_theme_support('wc-product-gallery-zoom');
	add_theme_support('wc-product-gallery-lightbox');
	add_theme_support('wc-product-gallery-slider');
}
add_action('after_setup_theme', 'tusenrosor_encoderit_woocommerce_setup');

/**
 * WooCommerce specific scripts & stylesheets.
 *
 * @return void
 */
function tusenrosor_encoderit_woocommerce_scripts()
{
	wp_enqueue_style('tusenrosor-encoderit-woocommerce-style', get_template_directory_uri() . '/woocommerce.css', array(), _S_VERSION);

	$font_path   = WC()->plugin_url() . '/assets/fonts/';
	$inline_font = '@font-face {
			font-family: "star";
			src: url("' . $font_path . 'star.eot");
			src: url("' . $font_path . 'star.eot?#iefix") format("embedded-opentype"),
				url("' . $font_path . 'star.woff") format("woff"),
				url("' . $font_path . 'star.ttf") format("truetype"),
				url("' . $font_path . 'star.svg#star") format("svg");
			font-weight: normal;
			font-style: normal;
		}';

	wp_add_inline_style('tusenrosor-encoderit-woocommerce-style', $inline_font);
	wp_enqueue_script('wc-cart-fragments');
	wp_enqueue_script('wc-cart');
}
add_action('wp_enqueue_scripts', 'tusenrosor_encoderit_woocommerce_scripts');

/**
 * Disable the default WooCommerce stylesheet.
 *
 * Removing the default WooCommerce stylesheet and enqueing your own will
 * protect you during WooCommerce core updates.
 *
 * @link https://docs.woocommerce.com/document/disable-the-default-stylesheet/
 */
add_filter('woocommerce_enqueue_styles', '__return_empty_array');

/**
 * Add 'woocommerce-active' class to the body tag.
 *
 * @param  array $classes CSS classes applied to the body tag.
 * @return array $classes modified to include 'woocommerce-active' class.
 */
function tusenrosor_encoderit_woocommerce_active_body_class($classes)
{
	$classes[] = 'woocommerce-active';

	return $classes;
}
add_filter('body_class', 'tusenrosor_encoderit_woocommerce_active_body_class');

/**
 * Related Products Args.
 *
 * @param array $args related products args.
 * @return array $args related products args.
 */
function tusenrosor_encoderit_woocommerce_related_products_args($args)
{
	$defaults = array(
		'posts_per_page' => 12,
		'columns'        => 4,
	);

	$args = wp_parse_args($defaults, $args);

	return $args;
}
add_filter('woocommerce_output_related_products_args', 'tusenrosor_encoderit_woocommerce_related_products_args');

/**
 * Remove default WooCommerce wrapper.
 */
remove_action('woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10);
remove_action('woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10);

if (! function_exists('tusenrosor_encoderit_woocommerce_wrapper_before')) {
	/**
	 * Before Content.
	 *
	 * Wraps all WooCommerce content in wrappers which match the theme markup.
	 *
	 * @return void
	 */
	function tusenrosor_encoderit_woocommerce_wrapper_before()
	{
?>
		<main id="primary" class="site-main">
		<?php
	}
}
add_action('woocommerce_before_main_content', 'tusenrosor_encoderit_woocommerce_wrapper_before');

if (! function_exists('tusenrosor_encoderit_woocommerce_wrapper_after')) {
	/**
	 * After Content.
	 *
	 * Closes the wrapping divs.
	 *
	 * @return void
	 */
	function tusenrosor_encoderit_woocommerce_wrapper_after()
	{
		?>
		</main><!-- #main -->
	<?php
	}
}
add_action('woocommerce_after_main_content', 'tusenrosor_encoderit_woocommerce_wrapper_after');

/**
 * Sample implementation of the WooCommerce Mini Cart.
 *
 * You can add the WooCommerce Mini Cart to header.php like so ...
 *
	<?php
		if ( function_exists( 'tusenrosor_encoderit_woocommerce_header_cart' ) ) {
			tusenrosor_encoderit_woocommerce_header_cart();
		}
	?>
 */

if (! function_exists('tusenrosor_encoderit_woocommerce_cart_link_fragment')) {
	/**
	 * Cart Fragments.
	 *
	 * Ensure cart contents update when products are added to the cart via AJAX.
	 *
	 * @param array $fragments Fragments to refresh via AJAX.
	 * @return array Fragments to refresh via AJAX.
	 */
	function tusenrosor_encoderit_woocommerce_cart_link_fragment($fragments)
	{
		ob_start();
		tusenrosor_encoderit_woocommerce_cart_link();
		$fragments['a.cart-contents'] = ob_get_clean();

		return $fragments;
	}
}


if (! function_exists('tusenrosor_encoderit_woocommerce_header_cart')) {
	/**
	 * Display Header Cart.
	 *
	 * @return void
	 */
	function tusenrosor_encoderit_woocommerce_header_cart()
	{
		if (is_cart()) {
			$class = 'current-menu-item';
		} else {
			$class = '';
		}
	?>
		<ul id="site-header-cart" class="site-header-cart">
			<li class="<?php echo esc_attr($class); ?>">
				<?php tusenrosor_encoderit_woocommerce_cart_link(); ?>
			</li>
			<li>
				<?php
				$instance = array(
					'title' => '',
				);

				the_widget('WC_Widget_Cart', $instance);
				?>
			</li>
		</ul>
	<?php
	}
}

//category hook
//price hook
add_filter('woocommerce_currency_symbol', function ($symbol, $currency) {
	return ' ' . $symbol;
}, 10, 2);
//api search hook
add_action('rest_api_init', function () {
	register_rest_route('tusenrosor/v1', '/search', [
		'methods'  => 'GET',
		'callback' => 'tusenrosor_ajax_search',
		'permission_callback' => '__return_true',
	]);
});

//color attibute image
// Add image upload field to Add/Edit term 
add_action('pa_color_add_form_fields', 'custom_pa_color_add_field', 10, 2);
function custom_pa_color_add_field($taxonomy)
{
	?>
	<div class="form-field term-group">
		<label for="term-image-id"><?php _e('Color Image', 'woocommerce'); ?></label>
		<input type="hidden" id="term-image-id" name="term-image-id" value="">
		<div id="term-image-wrapper"></div>
		<button type="button" class="upload_image_button button"><?php _e('Upload/Add Image', 'woocommerce'); ?></button>
		<button type="button" class="remove_image_button button"><?php _e('Remove Image', 'woocommerce'); ?></button>
	</div>
<?php
}

add_action('pa_color_edit_form_fields', 'custom_pa_color_edit_field', 10, 2);
function custom_pa_color_edit_field($term, $taxonomy)
{
	$image_id = get_term_meta($term->term_id, 'term_image_id', true);
?>
	<tr class="form-field term-group-wrap">
		<th scope="row"><label for="term-image-id"><?php _e('Color Image', 'woocommerce'); ?></label></th>
		<td>
			<input type="hidden" id="term-image-id" name="term-image-id" value="<?php echo esc_attr($image_id); ?>">
			<div id="term-image-wrapper">
				<?php if ($image_id) echo wp_get_attachment_image($image_id, 'thumbnail'); ?>
			</div>
			<button type="button" class="upload_image_button button"><?php _e('Upload/Add Image', 'woocommerce'); ?></button>
			<button type="button" class="remove_image_button button"><?php _e('Remove Image', 'woocommerce'); ?></button>
		</td>
	</tr>
<?php
}
add_action('created_pa_color', 'custom_save_pa_color_image', 10, 2);
add_action('edited_pa_color', 'custom_save_pa_color_image', 10, 2);
function custom_save_pa_color_image($term_id, $tt_id)
{
	if (isset($_POST['term-image-id']) && '' !== $_POST['term-image-id']) {
		update_term_meta($term_id, 'term_image_id', absint($_POST['term-image-id']));
	} else {
		delete_term_meta($term_id, 'term_image_id');
	}
}

//pa box
add_action('pa_box_add_form_fields', 'custom_pa_box_add_field', 10, 2);
function custom_pa_box_add_field($taxonomy)
{
?>
	<div class="form-field term-group">
		<label for="term-image-id"><?php _e('Color Image', 'woocommerce'); ?></label>
		<input type="hidden" id="term-image-id" name="term-image-id" value="">
		<div id="term-image-wrapper"></div>
		<button type="button" class="upload_image_button button"><?php _e('Upload/Add Image', 'woocommerce'); ?></button>
		<button type="button" class="remove_image_button button"><?php _e('Remove Image', 'woocommerce'); ?></button>
	</div>
<?php
}

add_action('pa_box_edit_form_fields', 'custom_pa_box_edit_field', 10, 2);
function custom_pa_box_edit_field($term, $taxonomy)
{
	$image_id = get_term_meta($term->term_id, 'term_image_id', true);
?>
	<tr class="form-field term-group-wrap">
		<th scope="row"><label for="term-image-id"><?php _e('Color Image', 'woocommerce'); ?></label></th>
		<td>
			<input type="hidden" id="term-image-id" name="term-image-id" value="<?php echo esc_attr($image_id); ?>">
			<div id="term-image-wrapper">
				<?php if ($image_id) echo wp_get_attachment_image($image_id, 'thumbnail'); ?>
			</div>
			<button type="button" class="upload_image_button button"><?php _e('Upload/Add Image', 'woocommerce'); ?></button>
			<button type="button" class="remove_image_button button"><?php _e('Remove Image', 'woocommerce'); ?></button>
		</td>
	</tr>
<?php
}
add_action('created_pa_box', 'custom_save_pa_box_image', 10, 2);
add_action('edited_pa_box', 'custom_save_pa_box_image', 10, 2);
function custom_save_pa_box_image($term_id, $tt_id)
{
	if (isset($_POST['term-image-id']) && '' !== $_POST['term-image-id']) {
		update_term_meta($term_id, 'term_image_id', absint($_POST['term-image-id']));
	} else {
		delete_term_meta($term_id, 'term_image_id');
	}
}
//pa box
add_action('admin_enqueue_scripts', function () {

	wp_enqueue_media();
	wp_enqueue_script('jquery');
	if (isset($_GET['taxonomy']) && strpos($_GET['taxonomy'], 'pa_') === 0) {
		wp_enqueue_media();
		wp_enqueue_script(
			'term-media-upload',
			get_template_directory_uri() . '/admin/assets/js/term-media-upload.js',
			['jquery'],
			'1.0',
			true
		);
	}
});
//variation hook
/**
 * Convert variation dropdowns to custom Tailwind radio buttons
 * while keeping WooCommerce AJAX variation logic working.
 */
add_filter('woocommerce_dropdown_variation_attribute_options_html', 'convert_variations_to_radio', 20, 2);
function convert_variations_to_radio($html, $args)
{
	// Get available options
	$options = $args['options'];
	$product = $args['product'];
	$attribute = $args['attribute'];
	$name = sanitize_title($attribute);
	$selected = isset($args['selected']) ? $args['selected'] : '';
	$radios = '';
	if (empty($options) && ! empty($product) && ! empty($attribute)) {
		$attributes = $product->get_variation_attributes();
		$options = $attributes[$attribute];
	}
	if (! empty($options)) {
		$radios .= '<div class="wc-variation-radios flex flex-wrap" data-attribute_name="attribute_' . esc_attr($name) . '">';
		foreach ($options as $option) {
			// Get term name (for labels)
			$term = get_term_by('slug', $option, $attribute);
			$label = $term ? $term->name : $option;
			$img_id   = $term ? get_term_meta($term->term_id, 'term_image_id', true) : '';
			$img_url  = $img_id ? wp_get_attachment_url($img_id) : wc_placeholder_img_src();
			$checked = checked(sanitize_title($selected), sanitize_title($option), false);
			$isActive = $checked ? 'active' : '';
			$radios .= sprintf(
				'<label class="[&.active]:border-(--theme-color) has-checked:border-(--theme-color) hover:border-(--theme-color) m-[5px] cursor-pointer flex items-center justify-center h-12 min-h-12 w-12 min-w-12 border border-[#EBEBEB] rounded-full duration-400 bg-white bg-no-repeat bg-center bg-cover %5$s" style="background-image:url(%4$s)"><input type="radio" class="hidden" name="attribute_%1$s" value="%2$s" %3$s></label> ',
				esc_attr($name),
				esc_attr($option),
				$checked,
				esc_url($img_url),
				esc_attr($isActive)
			);
		}
		$radios .= '</div>';
	}
	// Add hidden select to keep JS compatibility
	$radios .= '<select name="attribute_' . esc_attr($name) . '" style="display:none;">';
	foreach ($options as $option) {
		$selected_attr = selected($selected, $option, false);
		$radios .= '<option value="' . esc_attr($option) . '" ' . $selected_attr . '>' . esc_html($option) . '</option>';
	}
	$radios .= '</select>';
	return $radios;
}
// :brain: Keep JS syncing (update hidden select on click)
add_action('wp_footer', function () {
?>
	<script>
		document.addEventListener('DOMContentLoaded', function() {
			const radios = document.querySelectorAll('.wc-variation-radios input[type=radio]');
			radios.forEach(function(el) {
				// Add 'selected' class if this radio is already checked (default)
				if (el.checked) {
					const label = el.closest('label');
					if (label) {
						label.classList.add('selected');
					}
				}
				// Add change event
				el.addEventListener('change', function() {
					const container = el.closest('.wc-variation-radios');
					// Remove previous active class from all labels in this group
					container.querySelectorAll('label').forEach(label => label.classList.remove('selected'));
					// Add active class to the current label
					const label = el.closest('label');
					if (label) {
						label.classList.add('selected');
					}
					// Update hidden select
					const select = container.nextElementSibling;
					if (select) {
						select.value = el.value;
						select.dispatchEvent(new Event('change', {
							bubbles: true
						}));
					}
				});
			});
		});
	</script>
<?php
}, 100);

//category hook
//product page functionality
function tusenrosor_find_product_link($cat_slug, $color_slug = '', $box_slug = '')
{
	$args = [
		'post_type'      => 'product',
		'post_status'    => 'publish',
		'posts_per_page' => 1,
		'tax_query'      => [
			'relation' => 'AND',
			[
				'taxonomy' => 'product_cat',
				'field'    => 'slug',
				'terms'    => $cat_slug,
			],
		],
	];
	if ($color_slug) {
		$args['tax_query'][] = [
			'taxonomy' => 'pa_color',
			'field'    => 'slug',
			'terms'    => $color_slug,
		];
	}
	if ($box_slug) {
		$args['tax_query'][] = [
			'taxonomy' => 'pa_box',
			'field'    => 'slug',
			'terms'    => $box_slug,
		];
	}

	$query = new WP_Query($args);
	if ($query->have_posts()) {
		$link = get_permalink($query->posts[0]->ID);
		wp_reset_postdata();
		return $link;
	}
	return '';
}
//add to cart 
add_action('wp_ajax_custom_add_to_cart', 'custom_add_to_cart');
add_action('wp_ajax_nopriv_custom_add_to_cart', 'custom_add_to_cart');

function custom_add_to_cart()
{
	$product_id = intval($_POST['product_id']);
	$gift_selected = $_POST['gift_selected'];
	$paper = sanitize_text_field($_POST['paper'] ?? '');
	$paper_price = floatval($_POST['paper_price'] ?? 0);
	$comment = sanitize_textarea_field($_POST['comment'] ?? '');
	//print_r($_POST);
	if (!$product_id) {
		wp_send_json_error('No product ID.');
	}
	$gift_product_id = intval($gift_selected ?? 0);
	// Add main product
	$cart_item_key = WC()->cart->add_to_cart($product_id, 1, 0, [], [
		'gift_selected_product' => $gift_selected,
		'paper' => $paper,
		'paper_price' => $paper_price,
		'comment' => $comment,
	]);

	if ($gift_selected) {
		// Add gift product (set your gift product ID here)
		$gift_product_id = $gift_selected; // change this to your actual gift product ID
		$gift_cart_key = WC()->cart->add_to_cart($gift_product_id, 1, 0, [], [
			'is_gift' => true,
			'linked_to' => $product_id,
		]);

		if ($gift_cart_key && isset(WC()->cart->cart_contents[$gift_cart_key])) {
			WC()->cart->cart_contents[$gift_cart_key]['data']->set_price(0);
		}
	}

	wp_send_json_success(['cart_item_key' => $cart_item_key]);
}
//qty inc dec
// ✅ Custom AJAX handler to update cart item quantity
add_action('wp_ajax_woocommerce_update_cart_item', 'custom_woocommerce_update_cart_item');
add_action('wp_ajax_nopriv_woocommerce_update_cart_item', 'custom_woocommerce_update_cart_item');

function custom_woocommerce_update_cart_item()
{
	// Validate data
	$cart_item_key = sanitize_text_field($_POST['cart_item_key'] ?? '');
	$quantity = intval($_POST['quantity'] ?? 0);

	if (!$cart_item_key || $quantity < 1) {
		wp_send_json_error(['message' => 'Invalid data']);
	}

	// Get WooCommerce cart
	if (!WC()->cart) {
		wp_send_json_error(['message' => 'Cart not initialized']);
	}

	// ✅ Update quantity
	WC()->cart->set_quantity($cart_item_key, $quantity, true);

	// ✅ Recalculate totals
	WC()->cart->calculate_totals();

	// ✅ Return success + new totals
	wp_send_json_success([
		'cart_total' => strip_tags(WC()->cart->get_cart_total()),
		'cart_count' => WC()->cart->get_cart_contents_count(),
	]);
}

//qty inc dec

//remove cart product
add_action('wp_ajax_custom_remove_cart_item', 'custom_remove_cart_item');
add_action('wp_ajax_nopriv_custom_remove_cart_item', 'custom_remove_cart_item');

function custom_remove_cart_item()
{
	if (empty($_POST['cart_item_key'])) {
		wp_send_json_error(['message' => __('Ingen produktnyckel angiven.', 'tusenrosor')]);
	}

	$cart_item_key = sanitize_text_field($_POST['cart_item_key']);
	$cart = WC()->cart;
	$cart_contents = $cart->get_cart();

	// 1️⃣ Find the main product to remove
	if (!isset($cart_contents[$cart_item_key])) {
		wp_send_json_error(['message' => __('Produkten kunde inte hittas i varukorgen.', 'tusenrosor')]);
	}

	$main_item = $cart_contents[$cart_item_key];
	$main_product_id = $main_item['product_id'];

	// 2️⃣ Find any linked gift item for this product
	foreach ($cart_contents as $key => $item) {
		if (!empty($item['is_gift']) && intval($item['linked_to']) === intval($main_product_id)) {
			$cart->remove_cart_item($key); // remove gift line
		}
	}

	// 3️⃣ Remove main product
	$cart->remove_cart_item($cart_item_key);
	$cart->calculate_totals();

	wp_send_json_success([
		'message' => __('Produkten (och eventuell gåva) har tagits bort.', 'tusenrosor'),
		'total'   => strip_tags($cart->get_cart_total()),
	]);
}

//order coupon
// 💸 Apply coupon via AJAX
// 💸 Improved AJAX coupon apply handler
add_action('wp_ajax_apply_coupon_code_ajax', 'apply_coupon_code_ajax');
add_action('wp_ajax_nopriv_apply_coupon_code_ajax', 'apply_coupon_code_ajax');

function apply_coupon_code_ajax()
{
	$code = sanitize_text_field($_POST['coupon_code'] ?? '');
	if (!$code) {
		wp_send_json_error(['message' => __('Ingen kupongkod angiven', 'woocommerce')]);
	}

	$cart = WC()->cart;
	if (!$cart) {
		wp_send_json_error(['message' => __('Varukorgen har inte initierats', 'woocommerce')]);
	}

	// ✅ Try to apply the coupon
	$result = $cart->apply_coupon($code);
	$cart->calculate_totals();

	// ✅ Detect invalid or rejected coupons
	$coupon = new WC_Coupon($code);
	if (!$coupon->is_valid() || is_wp_error($result)) {
		$error_message = is_wp_error($result)
			? $result->get_error_message()
			: __('Invalid coupon code.', 'woocommerce');

		// Clean notices to avoid duplicates later
		wc_clear_notices();

		wp_send_json_error(['message' => $error_message]);
	}

	// ✅ If valid, success
	wc_clear_notices();
	wp_send_json_success([
		'total' => strip_tags($cart->get_cart_total()),
		'message' => __('Kupongen har tillämpats!', 'woocommerce')
	]);
}

//order coupon
// Save order note in Woo session
add_action('wp_ajax_save_order_note_ajax', 'save_order_note_ajax');
add_action('wp_ajax_nopriv_save_order_note_ajax', 'save_order_note_ajax');

function save_order_note_ajax()
{
	$note = sanitize_textarea_field($_POST['order_note'] ?? '');
	WC()->session->set('custom_order_note', $note);
	wp_send_json_success(['message' => __('Anteckning sparad', 'tusenrosor')]);
}
add_filter('woocommerce_checkout_get_value', function ($value, $input) {
	if ($input === 'order_comments' && WC()->session) {
		$saved_note = WC()->session->get('custom_order_note');
		if (!empty($saved_note)) {
			return $saved_note;
		}
	}
	return $value;
}, 10, 2);
// ✅ Attach note to checkout order


add_filter('woocommerce_get_item_data', function ($item_data, $cart_item) {
	if (!empty($cart_item['paper'])) {
		$item_data[] = [
			'name' => __('Paper Type', 'tusenrosor'),
			'value' => esc_html($cart_item['paper']),
		];
	}
	if (!empty($cart_item['comment'])) {
		$item_data[] = [
			'name' => __('Comment', 'tusenrosor'),
			'value' => esc_html($cart_item['comment']),
		];
	}
	if (!empty($cart_item['gift_selected'])) {
		$item_data[] = [
			'name' => __('Gift Product', 'tusenrosor'),
			'value' => __('Yes', 'tusenrosoromain'),
		];
	}
	return $item_data;
}, 10, 2);
add_action('woocommerce_checkout_create_order_line_item', function ($item, $cart_item_key, $values, $order) {
	if (!empty($values['paper'])) {
		$item->add_meta_data(__('Paper', 'tusenrosor'), $values['paper']);
	}

	if (!empty($values['paper_price'])) {
		$item->add_meta_data(__('Paper price', 'tusenrosor'), $values['paper_price']);
	}

	if (!empty($values['comment'])) {
		$item->add_meta_data(__('Meddelande', 'tusenrosor'), $values['comment']);
	}

	// ✅ Properly link gift product
	if (!empty($values['is_gift']) && !empty($values['linked_to'])) {
		$item->add_meta_data('is_gift', 'yes');
		$item->add_meta_data('linked_to', intval($values['linked_to']));
	}
}, 10, 4);
add_action('woocommerce_before_calculate_totals', function ($cart) {
	if (is_admin() && !defined('DOING_AJAX')) {
		return;
	}

	foreach ($cart->get_cart() as $cart_item_key => $cart_item) {

		// 🟢 1. If this is a gift item — make it free
		if (!empty($cart_item['is_gift'])) {
			$cart_item['data']->set_price(0);
		}

		// 🟢 2. If main product has extra paper price — add it
		if (!empty($cart_item['paper_price'])) {
			$base_price = $cart_item['data']->get_price();
			$cart_item['data']->set_price($base_price + floatval($cart_item['paper_price']));
		}
	}
});
function tusenrosor_get_real_cart_count()
{
	$count = 0;

	foreach (WC()->cart->get_cart() as $item) {

		if (empty($item['is_gift'])) {
			$count += $item['quantity'];
		}
	}
	return $count;
}
// ✅ Add custom cart fragments (count + total)
add_filter('woocommerce_add_to_cart_fragments', function ($fragments) {
	ob_start();
	if (WC()->session) {
		WC()->session->get_session_data(); // reload latest session
	}
	$cart_count = tusenrosor_get_real_cart_count();
	if ($cart_count > 9) {
		$totalcart = '+9';
	} else {
		$totalcart = $cart_count;
	}
?>
	<span id="cart-count" class="bg-(--theme-maroon) rounded-full h-3 min-w-3 flex items-center justify-center absolute -top-0.5 -right-0.5 text-white text-center text-[8px] font-normal py-0.5 px-1">
		<?php echo $totalcart; ?>
	</span>
	<?php
	$fragments['#cart-count'] = ob_get_clean();
	// 🔹 Detect if current page is the main Cart page
	$is_cart_page = !empty($_COOKIE['wc_is_cart_page']) && $_COOKIE['wc_is_cart_page'] === '1';
	$is_checkout_page = !empty($_COOKIE['wc_is_checkout_page']) && $_COOKIE['wc_is_checkout_page'] === '1';
	// 🔹 Render Cart Items
	ob_start();
	wc_get_template('cart/cart-items.php');
	$item_template_html = ob_get_clean();

	// 🔹 Render Cart Totals
	ob_start();
	wc_get_template('cart/cart-total.php');
	$item_total_template_html = ob_get_clean();

	//


	if ($is_cart_page) {
		// 🔹 Render Cart Items
		ob_start();
		wc_get_template('cart/cart-items.php', ['wrapper_id' => 'PagecartItems']);
		$page_item_template_html = ob_get_clean();

		// 🔹 Render Cart Totals
		ob_start();
		wc_get_template('cart/cart-total.php', ['wrapper_id' => 'PagecartItemTotalBtn']);
		$page_item_total_template_html = ob_get_clean();
		$fragments['#PagecartItems'] = $page_item_template_html;
		$fragments['#PagecartItemTotalBtn'] = $page_item_total_template_html;
		$fragments['#cartItems'] = $item_template_html;
		$fragments['#cartItemTotalBtn'] = $item_total_template_html;
	} else {
		$fragments['#cartItems'] = $item_template_html;
		$fragments['#cartItemTotalBtn'] = $item_total_template_html;
	}
	if ($is_checkout_page) {
		ob_start();
		wc_get_template('checkout/cart-total.php');
		$fragments['#checkoutTotalsSection'] = ob_get_clean();
	}

	return $fragments;
});

//add to cart 

//review
add_action('template_redirect', function () {
	if (!isset($_POST['submit_review'])) return;

	$product_id = get_the_ID();
	$user = wp_get_current_user();

	if (!wc_customer_bought_product($user->user_email, $user->ID, $product_id)) {
		wc_add_notice('Endast kunder som köpt produkten kan lämna en recension.', 'error');
		return;
	}

	$comment_id = wp_insert_comment([
		'comment_post_ID' => $product_id,
		'comment_author' => $user->display_name,
		'comment_author_email' => $user->user_email,
		'comment_content' => sanitize_textarea_field($_POST['review']),
		'comment_type' => 'comment',
		'user_id' => $user->ID,
		'comment_approved' => 0, // Moderate manually
	]);

	if ($comment_id) {
		// Save rating
		if (isset($_POST['rating'])) {
			update_comment_meta($comment_id, 'rating', (float)($_POST['rating']));
			update_comment_meta($comment_id, 'comment_title', ($_POST['comment_title']));
		}

		// Handle uploaded images
		if (!empty($_FILES['review_images']['name'][0])) {
			// ✅ Load WP upload functions
			require_once(ABSPATH . 'wp-admin/includes/file.php');
			$uploaded_ids = [];
			foreach ($_FILES['review_images']['name'] as $key => $value) {
				$file = [
					'name' => $_FILES['review_images']['name'][$key],
					'type' => $_FILES['review_images']['type'][$key],
					'tmp_name' => $_FILES['review_images']['tmp_name'][$key],
					'error' => $_FILES['review_images']['error'][$key],
					'size' => $_FILES['review_images']['size'][$key],
				];
				$upload = wp_handle_upload($file, ['test_form' => false]);
				if (!isset($upload['error'])) {
					$attachment = [
						'post_mime_type' => $upload['type'],
						'post_title' => basename($upload['file']),
						'post_content' => '',
						'post_status' => 'inherit',
					];
					$attach_id = wp_insert_attachment($attachment, $upload['file']);
					$uploaded_ids[] = $attach_id;
				}
			}
			if (!empty($uploaded_ids)) {
				update_comment_meta($comment_id, 'review_images', $uploaded_ids);
			}
		}

		wc_add_notice('Tack för din recension! Den väntar på godkännande.', 'success');
		wp_redirect(get_permalink($product_id));
		exit;
	}
});
add_action('wp_ajax_load_more_reviews', 'theme_load_more_reviews');
add_action('wp_ajax_nopriv_load_more_reviews', 'theme_load_more_reviews');

function theme_load_more_reviews()
{
	check_ajax_referer('load_more_reviews', 'nonce');

	$product_id = intval($_POST['product_id']);
	$page = max(1, intval($_POST['page']));
	$order = ($_POST['order'] === 'asc') ? 'ASC' : 'DESC';
	$per_page = 12;

	$args = [
		'post_id' => $product_id,
		'status'  => 'approve',

		'orderby' => 'comment_date_gmt',
		'order'   => $order,
		'number'  => $per_page,
		'offset'  => ($page - 1) * $per_page,
	];

	$comments = get_comments($args);
	ob_start();

	if ($comments) {
		foreach ($comments as $comment) {

			$rating = intval(get_comment_meta($comment->comment_ID, 'rating', true));
			$images = get_comment_meta($comment->comment_ID, 'review_images', true);
			$verified = wc_review_is_from_verified_owner($comment->comment_ID);
			$comment_product_id = $comment->comment_post_ID;
			$comment_product = wc_get_product($product_id);
	?>
			<div
				class="w-full py-6 flex flex-col items-start lg:flex-row border-b border-[#EBEBEB] space-y-4 lg:space-y-0 lg:space-x-4">

				<!-- User Info -->
				<div class="w-full lg:w-[250px] min-w-[250px] flex items-center space-x-3">
					<div class="h-15 w-15 rounded-full overflow-hidden">
						<?php echo get_avatar($comment, 60); ?>
					</div>
					<div class="flex flex-col items-start">
						<h5 class="text-base font-bold leading-[165%]">
							<?php echo esc_html($comment->comment_author); ?>
						</h5>
						<p class="text-xs text-(--sub-text) leading-[165%]">

							<?php echo human_time_diff(strtotime($comment->comment_date), current_time('timestamp')); ?> sedan

						</p>
					</div>
				</div>

				<!-- Review Content -->
				<div class="w-full">
					<div class="space-y-3">

						<!-- Fixed Stars + Verified -->
						<div class="space-y-3 md:space-y-0 md:space-x-3 flex flex-col md:flex-row md:justify-between">
							<div class="flex items-center space-x-3">
								<div class="flex items-center space-x-0.5">
									<?php for ($i = 0; $i < 5; $i++) : ?>
										<?php if ($i < $rating) { ?>
											<svg width="18" height="16" viewBox="0 0 18 16" fill="none" xmlns="http://www.w3.org/2000/svg">
												<path d="M8.72727 0V12.56L3.33382 16L5.06182 9.99273L0 6.112L6.46255 5.84L8.72727 0ZM8.72727 0V12.56L14.1207 16L12.3927 9.99273L17.4545 6.112L10.992 5.84L8.72727 0Z" fill="#303030" />
											</svg>
										<?php } else { ?>
											<svg width="18" height="16" viewBox="0 0 22 20" fill="none" xmlns="http://www.w3.org/2000/svg">
												<path
													d="M10.9091 1.23639L13.4979 7.15097L13.6061 7.39426L13.8713 7.40586L20.2735 7.68361L15.1493 11.6691L14.9369 11.8325L15.0103 12.0946L16.6687 18.0992L11.1005 14.5637L10.9091 14.4419L10.7178 14.5637L5.14958 18.0992L6.80799 12.0946L6.88136 11.8325L6.66895 11.6691L1.54478 7.68361L7.94699 7.40586L8.21215 7.39426L8.32033 7.15097L10.9091 1.23639Z"
													stroke="#303030"
													stroke-width="1.5"
													stroke-linejoin="round" />
											</svg>
										<?php } ?>
									<?php endfor; ?>

								</div>


								<?php if ($verified): ?>
									<p class="text-xs text-(--sub-text) leading-[165%]"><?php echo esc_html(__('Verifierat köp', 'tusenrosor')) ?></p>
								<?php endif; ?>

							</div>

							<div>
								<?php
								if ($comment_product) {

									$product_name = $comment_product->get_name();
									$product_link = get_permalink($comment_product_id);
								?>
									<a class="underline text-sm text-(--sub-text) leading-[165%] hover:opacity-65 duration-400" href="<?php echo $product_link; ?>">
										<?php echo $product_name; ?>
									</a>
								<?php } ?>
							</div>
						</div>

						<!-- Dynamic Title + Text -->
						<div class="space-y-2">
							<?php if (get_comment_meta($comment->comment_ID, 'comment_title', true)) { ?>
								<h4 class="text-sm leading-[165%] font-bold">
									<?php echo esc_html(get_comment_meta($comment->comment_ID, 'comment_title', true)); ?>
								</h4>
							<?php } ?>
							<div class="text-sm text-(--sub-text) leading-[165%]">
								<?php echo esc_html($comment->comment_content); ?>
							</div>
						</div>

						<!-- Dynamic Product Images -->
						<?php if (!empty($images) && is_array($images)): ?>
							<div>
								<div class="-m-[5px]">
									<div class="flex flex-wrap">
										<?php foreach ($images as $img_id):
											$img_url = wp_get_attachment_image_url($img_id, 'thumbnail');
										?>
											<div class="h-20 w-20 min-w-20 m-[5px]">
												<img
													class="h-full w-full object-cover bg-(--bg2)"
													src="<?php echo $img_url; ?>"
													alt="Product Image">
											</div>
										<?php endforeach; ?>
									</div>
								</div>
							</div>
						<?php endif; ?>

					</div>
				</div>
			</div>
<?php
		}
	}

	$html = ob_get_clean();

	// Determine if more pages exist
	$total_comments = get_comments([
		'post_id' => $product_id,
		'status'  => 'approve',
		'type'    => 'review',
		'count'   => true,
	]);

	$has_more = ($total_comments > $page * $per_page);

	wp_send_json_success([
		'html'      => $html,
		'has_more'  => $has_more,
	]);
}

//my account 
add_filter('woocommerce_account_menu_items', function ($items) {
	unset($items['dashboard']); // Remove Dashboard
	unset($items['downloads']); // Remove Downloads
	return $items;
});
add_action('template_redirect', function () {
	if (is_account_page() && empty(WC()->query->get_current_endpoint())) {
		wp_safe_redirect(wc_get_account_endpoint_url('orders'));
		exit;
	}
});
//register method
/**
 * Handle custom "Full Name" and "Terms & Conditions" on Woo registration page
 */
add_action('woocommerce_register_post', function ($username, $email, $validation_errors) {

	// Only apply on the standalone registration page (not checkout)
	if (is_checkout()) {
		return;
	}

	// Validate full name
	if (empty($_POST['username'])) {
		$validation_errors->add('full_name_error', __('Ange ditt fullständiga namn.', 'tusenrosor'));
	}

	// Validate terms & conditions checkbox
	if (empty($_POST['terms_conditions'])) {
		$validation_errors->add('terms_conditions_error', __('Du måste godkänna användarvillkoren.', 'tusenrosor'));
	}
}, 10, 3);


/**
 * Save custom full_name meta + modify username/display_name
 */
add_action('woocommerce_created_customer', function ($customer_id) {

	// Only run on registration page, not checkout
	if (is_checkout()) {
		return;
	}

	if (!empty($_POST['username'])) {
		$full_name = sanitize_text_field($_POST['username']);

		// Save to user meta
		update_user_meta($customer_id, 'full_name', $full_name);

		// Generate username (remove spaces, lowercase)
		$user_login = strtolower(preg_replace('/\s+/', '', $full_name));

		// Ensure unique username
		$base = $user_login;
		$i = 1;
		while (username_exists($user_login)) {
			$user_login = $base . $i;
			$i++;
		}

		// Update username and display name
		wp_update_user([
			'ID'           => $customer_id,
			'user_login'   => $user_login,
			'display_name' => $full_name,
		]);
	}
}, 10, 1);
add_filter('woocommerce_login_redirect', function ($redirect, $user) {
	if (! empty($_GET['redirect_to'])) {
		return esc_url_raw($_GET['redirect_to']);
	}
	return $redirect;
}, 10, 2);
add_filter('woocommerce_checkout_fields', function ($fields) {

	// Your custom Tailwind-style class
	$custom_class = 'bg-white px-4 py-1 w-full border border-[#EBEBEB] min-h-12 h-12 text-base md:text-sm font-normal text-(--theme-color) leading-[165%] outline-0 focus:border-(--theme-color) duration-400 placeholder:text-(--sub-text)';
	//print_r($fields);
	// Loop through all checkout field groups (billing, shipping, order, account)
	foreach ($fields as $section_key => &$section_fields) {
		foreach ($section_fields as $field_key => &$field) {


			$field['placeholder'] = $field['label'];
			if (isset($field['class']) && is_array($field['class'])) {
				// Merge existing classes
				$field['class'][] = 'w-full'; // wrapper div class
			}
			$field['label'] = ''; // remove visible label
			// Add your custom input class
			$field['input_class'][] = $custom_class;
		}
	}
	if (isset($fields['billing']['billing_postcode'])) {
		$fields['billing']['billing_postcode']['class'][] = 'form-row-postcode';
	}
	if (isset($fields['billing']['billing_city'])) {
		$fields['billing']['billing_city']['class'][] = 'form-row-city';
	}
	// $fields['billing']['billing_state']['type'] = 'select';
	// $fields['shipping']['shipping_state']['type'] = 'select';
	// $fields['billing']['billing_state']['options'] = [];
	// $fields['shipping']['shipping_state']['options'] = [];
	return $fields;
});
//remove coupon
add_action('wp_ajax_tusenrosor_remove_coupon', 'tusenrosor_remove_coupon');
add_action('wp_ajax_nopriv_tusenrosor_remove_coupon', 'tusenrosor_remove_coupon');

function tusenrosor_remove_coupon()
{
	check_ajax_referer('remove_coupon', 'security');

	if (isset($_POST['coupon_code'])) {
		$coupon_code = sanitize_text_field($_POST['coupon_code']);

		if (WC()->cart && in_array($coupon_code, WC()->cart->get_applied_coupons(), true)) {
			WC()->cart->remove_coupon($coupon_code);
			WC()->cart->calculate_totals();
			WC()->cart->set_session();
		}

		wp_send_json_success([
			'removed' => $coupon_code,
			'fragments' => wc_get_template_html('checkout/cart-total.php')
		]);
	}

	wp_send_json_error(['message' => 'Coupon not found']);
}
//disabled ore shipiing
add_filter('woocommerce_cart_needs_shipping', '__return_true');
add_filter('woocommerce_cart_needs_shipping_address', '__return_true');
// add_action('wp_head', function () {
// 	echo '<style>.woocommerce-shipping-methods, #shipping_method, .woocommerce-shipping-totals { display:none !important; }</style>';
// });
add_filter('woocommerce_shipping_methods', function ($methods) {
	// Disable all core methods except our custom one
	return ['tusenrosor_custom_shipping' => 'Tusenrosor_Custom_Shipping_Method'];
});
add_action('woocommerce_checkout_create_order', function ($order) {
	if ($note = WC()->session->get('custom_order_note')) {
		$order->set_customer_note($note);
		WC()->session->__unset('custom_order_note');
	}
	// Get shipping info from the session
	$info = WC()->session->get('tusenrosor_shipping_info');
	if (empty($info)) return;

	// Update the order meta data
	$order->update_meta_data('_delivery_leveranstid', sanitize_text_field($info['leveranstid'] ?? ''));
	$order->update_meta_data('_delivery_method', sanitize_text_field($info['method'] ?? ''));
	$order->update_meta_data('_delivery_date', sanitize_text_field($info['selectedDate'] ?? ''));
	$order->update_meta_data('_delivery_price', floatval($info['price'] ?? 0));
	WC()->session->__unset('tusenrosor_shipping_info');

	//remember
	if (isset($_POST['remember_enable']) && $_POST['remember_enable'] == "1") {
		if (!empty($_POST['remember_phone'])) {
			$order->update_meta_data('_remember_phone', sanitize_text_field($_POST['remember_phone']));
		}
	}
});

//validation checkout
add_action('woocommerce_checkout_process', function () {

	if (isset($_POST['remember_enable']) && $_POST['remember_enable'] == "1") {

		$phone = isset($_POST['remember_phone']) ? sanitize_text_field($_POST['remember_phone']) : "";

		if (!preg_match('/^\+46[0-9]{6,12}$/', $phone)) {
			wc_add_notice("Mobilnummer matchar inte det förväntade formatet.", 'error');
		}
	}
});
/**
 * AJAX endpoint to sync shipping cookie → WooCommerce session immediately
 */
add_action('wp_ajax_tusenrosor_sync_shipping', 'tusenrosor_sync_shipping');
add_action('wp_ajax_nopriv_tusenrosor_sync_shipping', 'tusenrosor_sync_shipping');
function tusenrosor_sync_shipping()
{
	if (!isset($_POST['data'])) {
		wp_send_json_error(['msg' => 'No data sent']);
	}

	$data = json_decode(stripslashes($_POST['data']), true);

	if (empty($data)) {
		wp_send_json_error(['msg' => 'Invalid JSON']);
	}

	WC()->session->set('tusenrosor_shipping_info', [
		'method'       => sanitize_text_field($data['method'] ?? ''),
		'leveranstid' => sanitize_text_field($data['leveranstid'] ?? ''),
		'selectedDate' => sanitize_text_field($data['selectedDate'] ?? ''),
		'price'        => floatval($data['price'] ?? 0),
		'businessDays' => intval($data['businessDays'] ?? 0),
	]);
	WC()->cart->calculate_totals();
	// print_r([
	// 	'method'       => sanitize_text_field($data['method'] ?? ''),
	// 	'selectedDate' => sanitize_text_field($data['selectedDate'] ?? ''),
	// 	'price'        => floatval($data['price'] ?? 0),
	// 	'businessDays' => intval($data['businessDays'] ?? 0),
	// ]);
	wp_send_json_success(['msg' => 'Session updated']);
}

add_action('woocommerce_cart_calculate_fees', function ($cart) {
	if (is_admin() && !defined('DOING_AJAX')) return;

	$cookie_name = 'tusenrosor_shipping_info';
	$session_key = 'tusenrosor_shipping_info';

	// ✅ prefer session over cookie for persistence
	$info = WC()->session->get($session_key);

	if (empty($info) && !empty($_COOKIE[$cookie_name])) {
		$info = json_decode(stripslashes($_COOKIE[$cookie_name]), true);
		WC()->session->set($session_key, $info);
	}



	if (empty($info)) return;

	$price  = floatval($info['price'] ?? 0);
	$method = sanitize_text_field($info['method'] ?? '');
	$date   = sanitize_text_field($info['selectedDate'] ?? '');

	if ($price > 0) {
		$label = sprintf(__('Leverans (%s - %s)', 'tusenrosor'), $method, $date);
		$cart->add_fee($label, $price, true);
	} else {
		$label = sprintf(__('Leverans (%s - %s)', 'tusenrosor'), $method, $date);
		$cart->add_fee($label, 0, true);
	}
	//tip

	if (isset($_POST['post_data'])) {
		parse_str($_POST['post_data'], $post_data);
	} else {
		$post_data = $_POST;
	}
	//print_r($post_data);
	$enabled = WC()->session->get("tip_enabled");
	$amount  = WC()->session->get("tip_amount");

	if ($enabled && $amount > 0) {
		$cart->add_fee("Dricks", $amount, true);
	}
}, 20);
//tip
add_action("wp_ajax_save_tip_amount", "save_tip_amount_ajax");
add_action("wp_ajax_nopriv_save_tip_amount", "save_tip_amount_ajax");

function save_tip_amount_ajax()
{
	$enable = isset($_POST['tip_enable']) && $_POST['tip_enable'] == 1;
	$amount = isset($_POST['tip_amount']) ? floatval($_POST['tip_amount']) : 0;

	if ($enable && $amount > 0) {
		WC()->session->set("tip_enabled", true);
		WC()->session->set("tip_amount", $amount);
	} else {
		WC()->session->__unset("tip_enabled");
		WC()->session->__unset("tip_amount");
	}
	WC()->cart->calculate_totals();
	wp_send_json_success($amount);
}

//country switcher
add_action('wp_ajax_get_country_states', 'tusenrosor_get_country_states');
add_action('wp_ajax_nopriv_get_country_states', 'tusenrosor_get_country_states');

function tusenrosor_get_country_states()
{
	$country = sanitize_text_field($_POST['country'] ?? '');
	$wc_countries = new WC_Countries();
	$states = $wc_countries->get_states($country);

	wp_send_json_success(['states' => $states ?: []]);
}
//woo state
add_filter('woocommerce_states', function ($states) {
	$states['SE'] = [
		'AB' => 'Stockholms län',
		'AC' => 'Västerbottens län',
		'BD' => 'Norrbottens län',
		'C'  => 'Uppsala län',
		'D'  => 'Södermanlands län',
		'E'  => 'Östergötlands län',
		'F'  => 'Jönköpings län',
		'G'  => 'Kronobergs län',
		'H'  => 'Kalmar län',
		'I'  => 'Gotlands län',
		'K'  => 'Blekinge län',
		'M'  => 'Skåne län',
		'N'  => 'Hallands län',
		'O'  => 'Västra Götalands län',
		'S'  => 'Värmlands län',
		'T'  => 'Örebro län',
		'U'  => 'Västmanlands län',
		'W'  => 'Dalarnas län',
		'X'  => 'Gävleborgs län',
		'Y'  => 'Västernorrlands län',
		'Z'  => 'Jämtlands län',
	];
	return $states;
});
//meddelande 
add_action('wp_ajax_tusenrosor_update_cart_comment', 'tusenrosor_update_cart_comment');
add_action('wp_ajax_nopriv_tusenrosor_update_cart_comment', 'tusenrosor_update_cart_comment');

function tusenrosor_update_cart_comment()
{

	if (empty($_POST['cart_key'])) {
		wp_send_json_error(['msg' => 'Missing cart key']);
	}

	$cart_key = sanitize_text_field($_POST['cart_key']);
	$comment  = sanitize_textarea_field($_POST['comment'] ?? '');

	$cart = WC()->cart;
	if (!$cart) {
		wp_send_json_error(['msg' => 'Cart not found']);
	}

	$cart_item = $cart->get_cart_item($cart_key);
	if (!$cart_item) {
		wp_send_json_error(['msg' => 'Cart item not found']);
	}

	// ✅ Update comment in cart item meta
	$cart->cart_contents[$cart_key]['comment'] = $comment;

	// ✅ Persist the cart update
	WC()->cart->set_session();

	wp_send_json_success(['msg' => 'Comment saved']);
}

//order status add
function register_custom_order_statuses()
{

	// Register "Packaging"
	register_post_status('wc-packaging', array(
		'label' => __('Packaging', 'woocommerce'),
		'public' => true,
		'show_in_admin_all_list' => true,
		'show_in_admin_status_list' => true,
		'exclude_from_search' => false,
		'label_count' => _n_noop('Packaging <span class="count">(%s)</span>', 'Packaging <span class="count">(%s)</span>')
	));

	// Register "On the Way"
	register_post_status('wc-on-the-way', array(
		'label' => __('On the Way', 'woocommerce'),
		'public' => true,
		'show_in_admin_all_list' => true,
		'show_in_admin_status_list' => true,
		'exclude_from_search' => false,
		'label_count' => _n_noop('On the Way <span class="count">(%s)</span>', 'On the Way <span class="count">(%s)</span>')
	));
}
add_action('init', 'register_custom_order_statuses');

// Add to WooCommerce dropdown lists
function add_custom_statuses_to_wc($order_statuses)
{
	$new_statuses = array();

	foreach ($order_statuses as $key => $label) {
		$new_statuses[$key] = $label;

		if ('wc-processing' === $key) {
			$new_statuses['wc-packaging'] = __('Packaging', 'woocommerce');
			$new_statuses['wc-on-the-way'] = __('On the Way', 'woocommerce');
		}
	}

	return $new_statuses;
}
add_filter('wc_order_statuses', 'add_custom_statuses_to_wc');

function step_icon($step, $current_step, $base)
{
	// completed step → always Instance-1.svg
	if ($step <= $current_step) {
		return $base . 'Instance-1.svg';
	}

	// active → own icon
	// if ($step == $current_step) {
	//     return $base . "Instance-$step.svg";
	// }

	// upcoming → own icon
	return $base . "Instance-$step.svg";
}
add_action('woocommerce_checkout_order_processed', function ($order_id) {
	if (is_user_logged_in() && isset($_POST['remember_phone'])) {
		update_user_meta(get_current_user_id(), 'remember_phone', sanitize_text_field($_POST['remember_phone']));
	}
});
