<?php

/**
 * Get full post/page details by ID.
 *
 * @param int $post_id The post or page ID.
 * @return array|null Returns an associative array of post details or null if not found.
 */
function get_image_url_by_id($id)
{
    if ($id) {
        return wp_get_attachment_image_url($id, 'full');
    } else {
        return '';
    }
}
function get_image_html_by_id($id, $class)
{
    if ($id) {
        return  wp_get_attachment_image($id, 'full', false, [
            'class' => $class,
            'alt' => 'Image'
        ]);
    } else {
        return '';
    }
}
function get_post_details_by_id($post_id)
{
    // Get the post object
    $post = get_post($post_id);

    // If no post found, return null
    if (!$post) {
        return null;
    }

    // Build response array
    $post_details = array(
        'ID'        => $post->ID,
        'title'     => get_the_title($post->ID),
        'excerpt'   => has_excerpt($post->ID) ? get_the_excerpt($post->ID) : wp_trim_words(strip_shortcodes($post->post_content), 30, '...'),
        'link'      => get_permalink($post->ID),
        'content'   => apply_filters('the_content', $post->post_content),
        'author'    => get_the_author_meta('display_name', $post->post_author),
        'date'      => get_the_date('', $post->ID),
        'thumbnail_id' => get_post_thumbnail_id($post->ID),
        'thumbnail' => get_the_post_thumbnail_url($post->ID, 'full'),
        'status'    => $post->post_status,
        'type'      => $post->post_type,
    );

    return $post_details;
}
// Fetch all terms of a taxonomy if a taxonomy name is passed
function get_taxonomy_terms($taxonomy_name, $hide_empty = false)
{
    if (empty($taxonomy_name)) {
        return []; // Return empty array if taxonomy name is not provided
    }

    // Fetch terms for the given taxonomy
    $terms = get_terms([
        'taxonomy' => $taxonomy_name,
        'hide_empty' => $hide_empty,  // Fetch even the terms with no posts
    ]);

    if (is_wp_error($terms)) {
        return [];
    }

    return $terms; // Return the terms of the given taxonomy
}
// Fetch posts based on multiple meta keys and values
function get_filtered_posts($args = [])
{
    // Example usage: Fetching posts with multiple meta keys and values
    // $args = [
    //     'post_type' => 'post', // Post type
    //     'posts_per_page' => -1, // Fetch all posts (no pagination)
    //     'meta_keys' => ['meta_key_1', 'meta_key_2'], // Multiple meta keys
    //     'meta_values' => ['value_1', 'value_2'], // Corresponding meta values
    //     'taxonomy' => 'category', // Taxonomy to filter by
    //     'taxonomy_terms' => [1, 2], // Term IDs for the taxonomy
    //     'orderby' => 'date', // Order by date
    //     'order' => 'ASC', // Order in ascending
    // ];
    // $posts = get_filtered_posts($args);
    // Default arguments for WP_Query
    $defaults = [
        'post_type' => 'post', // Default post type
        'posts_per_page' => 10, // Default limit
        'paged' => 1, // Default current page
        'taxonomy' => '', // Default taxonomy filter
        'taxonomy_terms' => [], // Default terms filter for taxonomy
        'meta_query' => [], // Default empty meta query
        'orderby' => 'date', // Default orderby
        'order' => 'DESC', // Default order (DESC)
    ];

    // Merge user provided arguments with defaults
    $args = wp_parse_args($args, $defaults);

    // Prepare meta query if meta_key and meta_value are passed
    if (!empty($args['meta_keys']) && !empty($args['meta_values'])) {
        $meta_queries = [];
        foreach ($args['meta_keys'] as $index => $meta_key) {
            if (isset($args['meta_values'][$index])) {
                $meta_queries[] = [
                    'key' => $meta_key,
                    'value' => $args['meta_values'][$index],
                    'compare' => '=', // Default compare operator
                ];
            }
        }
        // Assign the prepared meta queries to the query args
        $args['meta_query'] = $meta_queries;
    }

    // Handle taxonomy filter
    if (!empty($args['taxonomy']) && !empty($args['taxonomy_terms'])) {
        $args['tax_query'] = [
            [
                'taxonomy' => $args['taxonomy'],
                'field' => 'id',
                'terms' => $args['taxonomy_terms'],
                'operator' => 'IN',
            ]
        ];
    }

    // Prepare the WP_Query arguments
    $query_args = [
        'post_type' => $args['post_type'],
        'posts_per_page' => $args['posts_per_page'],
        'paged' => $args['paged'],
        'meta_query' => isset($args['meta_query']) ? $args['meta_query'] : [],
        'tax_query' => isset($args['tax_query']) ? $args['tax_query'] : [],
        'orderby' => $args['orderby'],
        'order' => $args['order'],
    ];

    // Execute the query
    $query = new WP_Query($query_args);

    // Check if posts are found
    if ($query->have_posts()) {
        wp_reset_postdata();
        return $query->posts; // Return the posts
    }
    wp_reset_postdata();
    return []; // Return empty array if no posts found
}



add_filter('template_include', 'print_home_page_templates', 99);

function print_home_page_templates($template)
{
    if (is_front_page() || is_home()) {
        // Output template name in the page source code
        echo '<!-- Template being used for the home page: ' . basename($template) . ' -->';
    }

    return $template;
}

/**
 * Get latest blog posts (optionally filtered by category).
 * Always resets query state to avoid conflicts.
 *
 * @param int         $count     Number of posts to fetch (default 3)
 * @param string|int  $category  Category slug or ID (optional)
 * @return array                 Array of WP_Post objects
 */
function get_latest_blog_posts($count = 3, $category = null, $layout = 'horizontal')
{
    // Define query args
    $args = [
        'post_type'      => 'post',
        'posts_per_page' => intval($count),
        'post_status'    => 'publish',
        'orderby'        => 'date',
         'post__not_in'   => [get_the_ID()],
        'order'          => 'DESC',
        'no_found_rows'  => true, // performance optimization
    ];
   

    // 🟢 Category filter (slug or ID)
    if (!empty($category)) {
        if (is_numeric($category)) {
            $args['cat'] = intval($category);
        } else {
            $args['category_name'] = sanitize_title($category);
        }
    }

    // Run query safely
    $query = new WP_Query($args);
    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();
            $post = get_post(get_the_ID());
            if ($layout == 'horizontal') {
                get_template_part('template-parts/blog-grid', null, ['post' => $post]);
            } else {
                get_template_part('template-parts/blog-grid-vertical', null, ['post' => $post]);
            }
        }
    }
    wp_reset_postdata();
    wp_reset_query();
    return ob_get_clean();
}


//placeholderimg
function get_placeholder_imageId()
{
    $general_setting = get_theme_option_data('general_setting');
    if (!empty($general_setting) && $general_setting['placeholder_image']) {
        return $general_setting['placeholder_image'];
    }
    return '';
}
function get_initials($name)
{
    $parts = preg_split('/\s+/', trim($name)); // split by any spaces
    $initials = '';

    foreach ($parts as $p) {
        if (!empty($p)) {
            $initials .= strtoupper($p[0]);
        }
    }

    return $initials;
}
add_action( 'wp_head', function() {
     if ( is_front_page() ) {
    $front_page_id = get_option( 'page_on_front' );
    $hero_section=get_field('hero_banner',$front_page_id);
     $bannerimage = wp_get_attachment_image_url($hero_section['banner'], 'full');
    //print_r( $hero_Section);
    ?>
    <link rel="preload" as="image" href="<?php echo $bannerimage;?>" fetchpriority="high">
    <?php
     }
});