jQuery(document).ready(function ($) {

    const canvas = document.getElementById("mugCanvasCurved");
    const ctx = canvas.getContext("2d");

    // Mug Image
    let mug = new Image();
    mug.src = MugMockupCurved.mugImage;

    // Mug Mask
    let mask = new Image();
    mask.src = MugMockupCurved.mugMask;

    // User Uploaded Image
    let userImg = null;

    let size = 300;
    let curvature = 25;
    let alignment = "middle";
    let flipH = false, flipV = false;

    mug.onload = drawCanvas;
    mask.onload = drawCanvas;

    /** FULL FIX for Broken State Error */
    function safeDecode(img) {
        return new Promise((resolve) => {
            if (img.complete) {
                img.decode().then(resolve).catch(resolve);
            } else {
                img.onload = () => img.decode().then(resolve).catch(resolve);
            }
        });
    }

    async function drawCanvas() {

        await safeDecode(mug);
        await safeDecode(mask);

        ctx.clearRect(0, 0, canvas.width, canvas.height);

        /** Draw Mug Base */
        ctx.drawImage(mug, 0, 0, canvas.width, canvas.height);

        /** Mug Color Layer */
        let color = $("#mugColor").val();

        ctx.globalCompositeOperation = "multiply";
        ctx.fillStyle = color;
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        /** Restore gloss + shadows */
        ctx.globalCompositeOperation = "overlay";
        ctx.drawImage(mask, 0, 0, canvas.width, canvas.height);

        ctx.globalCompositeOperation = "source-over";

        /** Draw user image warped */
        if (userImg) {
            await safeDecode(userImg);

            let warped = warpImageToCylinder(userImg, size, curvature, flipH, flipV);

            let yOffset = (alignment === "top") ? 180 :
                          (alignment === "middle") ? 320 : 460;

            ctx.drawImage(warped, 260, yOffset);
        }
    }

    /** Upload Image Handler with decode fix */
    $("#uploadImageCurved").on("change", function (e) {

        const reader = new FileReader();

        reader.onload = function (ev) {
            userImg = new Image();

            userImg.onload = async function () {
                await safeDecode(userImg);
                drawCanvas();
            };

            userImg.src = ev.target.result;
        };

        reader.readAsDataURL(this.files[0]);
    });

    /** Controls */
    $("#curvedResize").on("input", function () {
        size = parseInt($(this).val()) * 3;
        drawCanvas();
    });

    $("#curveRange").on("input", function () {
        curvature = parseInt($(this).val());
        drawCanvas();
    });

    $(".align-btn").on("click", function () {
        alignment = $(this).data("align");
        drawCanvas();
    });

    $("#flipH").on("click", () => { flipH = !flipH; drawCanvas(); });
    $("#flipV").on("click", () => { flipV = !flipV; drawCanvas(); });

    $("#mugColor").on("input", drawCanvas);

    $("#saveCurved").on("click", function () {
        let img = canvas.toDataURL("image/png");
        let a = document.createElement("a");
        a.href = img;
        a.download = "curved-mug.png";
        a.click();
    });

    /** Cylinder Warp Function (No Import, WP Safe) */
    function warpImageToCylinder(img, size, curvature, flipH, flipV) {

        let off = document.createElement("canvas");
        let ctx2 = off.getContext("2d");

        off.width = size;
        off.height = size;

        let temp = document.createElement("canvas");
        let tctx = temp.getContext("2d");
        temp.width = size;
        temp.height = size;

        try {
            tctx.drawImage(img, 0, 0, size, size);
        } catch (e) {
            console.log("Warp skipped, image not ready:", e);
            return off;
        }

        let src = tctx.getImageData(0, 0, size, size);
        let dest = ctx2.createImageData(size, size);

        for (let y = 0; y < size; y++) {
            let offsetX = Math.sin((y / size) * Math.PI) * curvature;

            for (let x = 0; x < size; x++) {
                let newX = x + offsetX;

                if (flipH) newX = size - newX;
                let newY = flipV ? size - y : y;

                newX = Math.max(0, Math.min(size - 1, Math.floor(newX)));

                let srcIndex = (y * size + x) * 4;
                let destIndex = (newY * size + newX) * 4;

                dest.data[destIndex] = src.data[srcIndex];
                dest.data[destIndex + 1] = src.data[srcIndex + 1];
                dest.data[destIndex + 2] = src.data[srcIndex + 2];
                dest.data[destIndex + 3] = src.data[srcIndex + 3];
            }
        }

        ctx2.putImageData(dest, 0, 0);

        return off;
    }
});
