jQuery(document).ready(function($) {
    console.log('ACF Simple Repeater: JavaScript initialized');

    // Ensure acfSimpleRepeater is defined
    if (typeof acfSimpleRepeater === 'undefined') {
        console.error('ACF Simple Repeater: acfSimpleRepeater object is undefined');
        return;
    }

    // Add row
    $(document).on('click', '.acf-simple-repeater .acf-button[data-event="add-row"]', function(e) {
        e.preventDefault();
        console.log('ACF Simple Repeater: Add row clicked');
        var $repeater = $(this).closest('.acf-simple-repeater');
        var repeaterId = $repeater.data('id');
        var subFieldGroupId = $repeater.data('subfield-group');

        if (!subFieldGroupId) {
            console.error('ACF Simple Repeater: Subfield group ID is missing for repeater ' + repeaterId);
            return;
        }

        // AJAX request to get subfields
        console.log('ACF Simple Repeater: AJAX request for options, group_id=' + subFieldGroupId);
        $.ajax({
            url: acfSimpleRepeater.ajaxurl,
            type: 'POST',
            data: {
                action: 'acf_simple_repeater_get_options',
                nonce: acfSimpleRepeater.nonce,
                field_group_id: subFieldGroupId
            },
            success: function(response) {
                console.log('ACF Simple Repeater: AJAX response', response);
                if (!response.success || !response.data.subFields) {
                    console.error('ACF Simple Repeater: Failed to load subfields for group_id=' + subFieldGroupId, response.data.message);
                    return;
                }

                var $rows = $repeater.find('.acf-repeater-rows');
                var index = $rows.find('.acf-row').length;
                var $row = $('<div class="acf-row" data-index="' + index + '"></div>');

                $.each(response.data.subFields, function(i, subField) {
                    var $subField = $('<div class="acf-sub-field" data-key="' + subField.key + '"></div>');
                    $subField.append('<label>' + subField.label + '</label>');

                    switch (subField.type) {
                        case 'wysiwyg':  // Add specific handling for WYSIWYG fields
                            $subField.append('<div class="acf-repeater-field" data-key="' + subField.key + '">' +
                                '<textarea class="acf-repeater-field wysiwyg-editor" data-key="' + subField.key + '">' + (subField.value || '') + '</textarea>' +
                                '</div>');
                            break;
                        case 'text':
                        case 'email':
                        case 'url':
                        case 'number':
                            $subField.append('<input type="' + subField.type + '" class="acf-repeater-field" data-key="' + subField.key + '" placeholder="' + (subField.placeholder || '') + '"' + (subField.min ? ' min="' + subField.min + '"' : '') + '>');
                            break;
                        case 'textarea':
                            $subField.append('<textarea class="acf-repeater-field" data-key="' + subField.key + '" placeholder="' + (subField.placeholder || '') + '"></textarea>');
                            break;
                        case 'image':
                        case 'file':
                            $subField.append('<div class="acf-repeater-' + subField.type + '">' +
                                '<input type="hidden" class="acf-repeater-field" data-key="' + subField.key + '">' +
                                (subField.type === 'image' ? '<img src="" class="acf-repeater-image-preview" style="display:none; max-width:100px;">' : '<span class="acf-repeater-file-preview"></span>') +
                                '<button class="acf-button button" data-event="select-' + subField.type + '">' + (subField.type === 'image' ? 'Select Image' : 'Select File') + '</button>' +
                                '</div>');
                            break;
                        case 'select':
                        case 'radio':
                            var $select = $('<select class="acf-repeater-field" data-key="' + subField.key + '" data-type="' + subField.type + '"></select>');
                            $select.append('<option value="">Select an option</option>');
                            $.each(subField.choices, function(key, label) {
                                $select.append('<option value="' + key + '">' + label + '</option>');
                            });
                            $subField.append($select);
                            break;
                        case 'checkbox':
                            var $checkboxDiv = $('<div class="acf-repeater-field" data-key="' + subField.key + '" data-type="checkbox"></div>');
                            $.each(subField.choices, function(key, label) {
                                $checkboxDiv.append('<label><input type="checkbox" value="' + key + '"> ' + label + '</label>');
                            });
                            $subField.append($checkboxDiv);
                            break;
                        case 'true_false':
                            $subField.append('<input type="checkbox" class="acf-repeater-field" data-key="' + subField.key + '" value="1">');
                            break;
                        case 'date_picker':
                        case 'time_picker':
                        case 'date_time_picker':
                            $subField.append('<input type="text" class="acf-repeater-field" data-key="' + subField.key + '" data-type="' + subField.type + '">');
                            break;
                        case 'user':
                        case 'page_link':
                        case 'post_object':
                        case 'relationship':
                        case 'taxonomy':
                            var $select = $('<select class="acf-repeater-field" data-key="' + subField.key + '" data-type="' + subField.type + '"' + (['relationship', 'taxonomy'].includes(subField.type) ? ' multiple' : '') + '></select>');
                            $select.append('<option value="">Select an option</option>');
                            $.ajax({
                                url: acfSimpleRepeater.ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'acf_simple_repeater_get_options',
                                    nonce: acfSimpleRepeater.nonce,
                                    field_key: subField.key,
                                    field_type: subField.type,
                                    post_type: subField.post_type || '',
                                    taxonomy: subField.taxonomy || ''
                                },
                                success: function(optResponse) {
                                    console.log('ACF Simple Repeater: Options AJAX response for ' + subField.type, optResponse);
                                    if (optResponse.success && optResponse.data.options) {
                                        $.each(optResponse.data.options, function(key, label) {
                                            $select.append('<option value="' + key + '">' + label + '</option>');
                                        });
                                    }
                                    if (subField.type === 'date_picker' || subField.type === 'date_time_picker') {
                                        $subField.find('.acf-repeater-field[data-type="date_picker"], .acf-repeater-field[data-type="date_time_picker"]').datepicker({
                                            dateFormat: 'yy-mm-dd'
                                        });
                                    } else if (subField.type === 'time_picker') {
                                        $subField.find('.acf-repeater-field[data-type="time_picker"]').timepicker();
                                    }
                                },
                                error: function(xhr, status, error) {
                                    console.error('ACF Simple Repeater: Failed to load options for ' + subField.type, xhr.responseText);
                                }
                            });
                            $subField.append($select);
                            break;
                    }
                    $row.append($subField);
                });

                $row.append('<div class="acf-row-handle"><a class="acf-icon -minus" href="#" data-event="remove-row" title="Remove"></a></div>');
                $rows.append($row);

                // Initialize WYSIWYG editor (if present)
                $row.find('.wysiwyg-editor').each(function() {
                    if (typeof tinymce !== 'undefined') {
                        tinymce.init({
                            target: this,
                            inline: true,
                            menubar: false,
                            toolbar: 'bold italic underline',
                            plugins: 'link image'
                        });
                    }
                });
            },
            error: function(xhr, status, error) {
                console.error('ACF Simple Repeater: AJAX error for group_id=' + subFieldGroupId, xhr.responseText);
            }
        });
    });

    // Remove row
    $(document).on('click', '.acf-simple-repeater .acf-icon[data-event="remove-row"]', function(e) {
        e.preventDefault();
        console.log('ACF Simple Repeater: Remove row clicked');
        var $repeater = $(this).closest('.acf-simple-repeater');
        $(this).closest('.acf-row').remove();
        
        // After row removal, update hidden input
        updateRepeaterData($repeater);
    });
    $(document).on('click', '.acf-simple-repeater [data-event^="select-"]', function(e) {
        e.preventDefault();
        console.log('ACF Simple Repeater: Select ' + $(this).data('event') + ' clicked');
        var $button = $(this);
        var $subField = $button.closest('.acf-sub-field');
        var type = $button.data('event').replace('select-', '');
        var frame = wp.media({
            title: 'Select ' + type.charAt(0).toUpperCase() + type.slice(1),
            button: { text: 'Select' },
            multiple: false,
            library: { type: type }
        });

        frame.on('select', function() {
            var attachment = frame.state().get('selection').first().toJSON();
            $subField.find('.acf-repeater-field[data-key]').val(attachment.id);
            if (type === 'image') {
                $subField.find('.acf-repeater-image-preview').attr('src', attachment.url).show();
            } else {
                $subField.find('.acf-repeater-file-preview').text(attachment.filename);
            }
            updateRepeaterData($subField.closest('.acf-simple-repeater'));
        });

        frame.open();
    });
    // Update repeater data
    function updateRepeaterData($repeater) {
        console.log('ACF Simple Repeater: Updating repeater data');
        var data = [];
        $repeater.find('.acf-row').each(function() {
            var $row = $(this);
            var rowData = {};
            $row.find('.acf-repeater-field').each(function() {
                var $field = $(this);
                var key = $field.data('key');
                var type = $field.data('type') || $field.attr('type') || 'text';
                var value;

                // Handle WYSIWYG (textarea) field
                if ($field.hasClass('wysiwyg-editor')) {
                    value = $field.val();  // Get the value of the editor field
                } else if (type === 'checkbox') {
                    value = [];
                    $field.find('input:checked').each(function() {
                        value.push($(this).val());
                    });
                } else if (type === 'select' && $field.prop('multiple')) {
                    value = $field.val() || [];
                } else {
                    value = $field.val();
                }
                rowData[key] = value;
            });
            data.push(rowData);
        });

        // If no rows, clear the hidden input field
        if (data.length === 0) {
            $repeater.find('.acf-simple-repeater-data').val('');
        } else {
            $repeater.find('.acf-simple-repeater-data').val(JSON.stringify(data));
        }
    }

    // Update data on input change
    $(document).on('change keyup', '.acf-simple-repeater .acf-repeater-field', function() {
        console.log('ACF Simple Repeater: Field changed, updating data');
        updateRepeaterData($(this).closest('.acf-simple-repeater'));
    });
});
