<?php
/*
Plugin Name: ACF Simple Repeater
Description: Mimics ACF Pro's repeater field in ACF Free using field name hooks, hiding text field input
Version: 1.5.5
Author: Your Name
Text Domain: acf-simple-repeater
*/

if (!defined('ABSPATH')) {
    exit;
}
if (!function_exists('is_plugin_active') && is_admin()) {
    include_once ABSPATH . 'wp-admin/includes/plugin.php';
}
interface FieldRendererInterface
{
    public function render_field($field);
}

class ACFSimpleRepeaterPlugin
{
    private $field_renderer;

    public function __construct()
    {
        $this->field_renderer = new ACFSimpleRepeaterField();
    }

    public function init()
    {
        include_once(ABSPATH . 'wp-admin/includes/plugin.php');
        if (is_plugin_active('advanced-custom-fields/acf.php')) {
            add_action('acf/render_field_settings/type=text', [$this, 'render_field_settings'], 10, 1);
            add_action('acf/input/admin_enqueue_scripts', [$this->field_renderer, 'input_admin_enqueue_scripts']);
            add_action('wp_ajax_acf_simple_repeater_get_options', [$this->field_renderer, 'ajax_get_options']);
            add_action('wp_ajax_nopriv_acf_simple_repeater_get_options', [$this->field_renderer, 'ajax_get_options']);
            add_action('acf/init', [$this, 'register_repeater_hooks']);
        } else {

            add_action('admin_notices', [$this, 'acf_missing_notice']);
        }
    }

    public function acf_missing_notice()
    {
?>
        <div class="notice notice-error">
            <p><?php _e('ACF Simple Repeater requires ACF to be installed and active.', 'acf-simple-repeater'); ?></p>
        </div>
    <?php
    }

    public function render_field_settings($field)
    {
        // Add "Is Repeater" toggle
        acf_render_field_setting($field, [
            'label' => __('Is Repeater', 'acf-simple-repeater'),
            'instructions' => __('Enable to use this field as a repeater.', 'acf-simple-repeater'),
            'name' => 'is_repeater',
            'type' => 'true_false',
            'ui' => 1
        ]);
        // Add subfield group ID setting
        acf_render_field_setting($field, [
            'label' => __('Repeater Subfield Group ID', 'acf-simple-repeater'),
            'instructions' => __('Enter the ACF field group ID (e.g., group_6727f12345678) containing the subfields to repeat. Required if "Is Repeater" is enabled.', 'acf-simple-repeater'),
            'name' => 'repeater_subfield_group',
            'type' => 'text',
            'placeholder' => 'e.g., group_6727f12345678',
            'conditional_logic' => [
                [
                    [
                        'field' => 'is_repeater',
                        'operator' => '==',
                        'value' => '1'
                    ]
                ]
            ]
        ]);
    }

    public function register_repeater_hooks()
    {
        $field_groups = acf_get_field_groups();
        foreach ($field_groups as $group) {
            $fields = acf_get_fields($group['key']);
            foreach ($fields as $field) {
                if ($field['type'] === 'text' && !empty($field['is_repeater'])) {
                    add_action('acf/render_field/name=' . $field['name'], [$this->field_renderer, 'render_field'], 10, 1);
                }
            }
        }
    }
}

class ACFSimpleRepeaterField implements FieldRendererInterface
{
    public function render_field($field)
    {
        // If the repeater field is not active or doesn't have subfields, render it as a standard text field
        if (empty($field['is_repeater']) || empty($field['repeater_subfield_group'])) {
            echo '<div class="acf-input">';
            acf_render_field(['type' => 'text', 'name' => $field['name'], 'value' => $field['value']]);
            echo '</div>';
            return;
        }

        // Get the value of the repeater field, default to empty array if not set
        $value = $field['value'] ? json_decode($field['value'], true) : [];
        $value = is_array($value) ? $value : [];

        // If there are no rows, set the value to empty to save empty data
        if (empty($value)) {
            $value = null;  // ACF should save null when there are no rows
        }

        // Get subfields of the repeater
        $sub_field_group_id = $field['repeater_subfield_group'];
        $sub_fields = $this->get_sub_fields($sub_field_group_id);

        // If no subfields are found, display an error
        if (empty($sub_fields)) {
            echo '<div class="notice notice-error"><p>' . __('No subfields found in the specified field group: ', 'acf-simple-repeater') . esc_html($sub_field_group_id) . '</p></div>';
            return;
        }

        // If there are no rows (empty repeater), don't display the field


        $id = uniqid('acf_simple_repeater_');
    ?>
        <div class="acf-simple-repeater" data-id="<?php echo esc_attr($id); ?>" data-subfield-group="<?php echo esc_attr($sub_field_group_id); ?>">

            <div class="acf-repeater-rows">
                <input type="hidden" name="<?php echo esc_attr($field['name']); ?>" class="acf-simple-repeater-data updates" value="<?php echo esc_attr(json_encode($value)); ?>">
                <?php if (!empty($value)) {
                    foreach ($value as $i => $row): ?>

                        <div class="acf-row" data-index="<?php echo esc_attr($i); ?>">
                            <?php foreach ($sub_fields as $sub_field): ?>
                                <div class="acf-sub-field" data-key="<?php echo esc_attr($sub_field['key']); ?>">
                                    <label><?php echo esc_html($sub_field['label']); ?></label>
                                    <?php $this->render_sub_field($sub_field, $row, $i); ?>
                                </div>
                            <?php endforeach; ?>
                            <div class="acf-row-handle">
                                <a class="acf-icon -minus" href="#" data-event="remove-row" title="<?php _e('Remove', 'acf-simple-repeater'); ?>"></a>
                            </div>
                        </div>
                <?php endforeach;
                } ?>
            </div>
            <div class="acf-actions">
                <a class="acf-button button button-primary" href="#" data-event="add-row"><?php _e('Add Row', 'acf-simple-repeater'); ?></a>
            </div>
        </div>
        <?php
    }

    private function get_sub_fields($sub_field_group_id)
    {
        // Validate group ID format
        if (!preg_match('/^group_[a-z0-9_-]+$/', $sub_field_group_id)) {

            return [];
        }

        $field_group = acf_get_field_group($sub_field_group_id);
        if (!$field_group) {

            return [];
        }

        $sub_fields = acf_get_fields($sub_field_group_id);
        if (!$sub_fields) {

            return [];
        }

        // Filter supported field types
        return array_filter($sub_fields, function ($field) {
            return in_array($field['type'], [
                'text',
                'textarea',
                'number',
                'email',
                'url',
                'image',
                'file',
                'select',
                'checkbox',
                'radio',
                'true_false',
                'date_picker',
                'time_picker',
                'date_time_picker',
                'user',
                'page_link',
                'post_object',
                'relationship',
                'taxonomy'
            ]);
        });
    }

    private function render_sub_field($sub_field, $row, $index)
    {
        $value = isset($row[$sub_field['key']]) ? $row[$sub_field['key']] : '';
        switch ($sub_field['type']) {
            case 'text':
            case 'email':
            case 'url':
            case 'number':
        ?>
                <input type="<?php echo esc_attr($sub_field['type']); ?>" class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" value="<?php echo esc_attr($value); ?>" placeholder="<?php echo esc_attr($sub_field['placeholder'] ?? ''); ?>" <?php echo isset($sub_field['min']) ? 'min="' . esc_attr($sub_field['min']) . '"' : ''; ?>>
            <?php
                break;
            case 'textarea':
            ?>
                <textarea class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" placeholder="<?php echo esc_attr($sub_field['placeholder'] ?? ''); ?>"><?php echo esc_textarea($value); ?></textarea>
            <?php
                break;
            case 'wysiwyg': // Editor (WYSIWYG)
            ?>
                <div class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>">
                    <?php
                    // Use ACF to render the editor field
                    acf_render_field([
                        'type'  => 'wysiwyg',
                        'name'  => $sub_field['key'],
                        'value' => $value,
                        'field' => $sub_field,
                    ]);
                    ?>
                </div>
            <?php
                break;
            case 'image':
            case 'file':
                $preview_url = $value ? wp_get_attachment_url($value) : '';
                $preview_style = $sub_field['type'] === 'image' ? 'max-width: 100px;' : '';
            ?>
                <div class="acf-repeater-<?php echo esc_attr($sub_field['type']); ?>">
                    <input type="hidden" class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" value="<?php echo esc_attr($value); ?>">
                    <?php if ($sub_field['type'] === 'image'): ?>
                        <img src="<?php echo esc_url($preview_url); ?>" class="acf-repeater-image-preview" style="<?php echo $preview_url ? $preview_style : 'display:none;'; ?>">
                    <?php else: ?>
                        <span class="acf-repeater-file-preview"><?php echo $preview_url ? esc_html(basename($preview_url)) : ''; ?></span>
                    <?php endif; ?>
                    <button class="acf-button button" data-event="select-<?php echo esc_attr($sub_field['type']); ?>"><?php echo $sub_field['type'] === 'image' ? __('Select Image', 'acf-simple-repeater') : __('Select File', 'acf-simple-repeater'); ?></button>
                </div>
            <?php
                break;
            case 'select':
            case 'radio':
            ?>
                <select class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" data-type="<?php echo esc_attr($sub_field['type']); ?>">
                    <option value=""><?php _e('Select an option', 'acf-simple-repeater'); ?></option>
                    <?php foreach ($sub_field['choices'] as $key => $label): ?>
                        <option value="<?php echo esc_attr($key); ?>" <?php selected($value, $key); ?>><?php echo esc_html($label); ?></option>
                    <?php endforeach; ?>
                </select>
            <?php
                break;
            case 'checkbox':
            ?>
                <div class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" data-type="checkbox">
                    <?php foreach ($sub_field['choices'] as $key => $label): ?>
                        <label><input type="checkbox" value="<?php echo esc_attr($key); ?>" <?php echo is_array($value) && in_array($key, $value) ? 'checked' : ''; ?>> <?php echo esc_html($label); ?></label>
                    <?php endforeach; ?>
                </div>
            <?php
                break;
            case 'true_false':
            ?>
                <input type="checkbox" class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" value="1" <?php checked($value, 1); ?>>
            <?php
                break;
            case 'date_picker':
            case 'time_picker':
            case 'date_time_picker':
            ?>
                <input type="text" class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" value="<?php echo esc_attr($value); ?>" data-type="<?php echo esc_attr($sub_field['type']); ?>">
            <?php
                break;
            case 'user':
            case 'page_link':
            case 'post_object':
            case 'relationship':
            case 'taxonomy':
            ?>
                <select class="acf-repeater-field" data-key="<?php echo esc_attr($sub_field['key']); ?>" data-type="<?php echo esc_attr($sub_field['type']); ?>" <?php echo in_array($sub_field['type'], ['relationship', 'taxonomy']) ? 'multiple' : ''; ?>>
                    <option value=""><?php _e('Select an option', 'acf-simple-repeater'); ?></option>
                    <?php
                    $options = $this->get_select_options($sub_field, $value);
                    foreach ($options as $key => $label): ?>
                        <option value="<?php echo esc_attr($key); ?>" <?php echo in_array($sub_field['type'], ['relationship', 'taxonomy']) && is_array($value) && in_array($key, $value) ? 'selected' : selected($value, $key); ?>><?php echo esc_html($label); ?></option>
                    <?php endforeach; ?>
                </select>
            <?php
                break;
            default:
            ?>
                <p><?php _e('Unsupported field type: ', 'acf-simple-repeater');
                    echo esc_html($sub_field['type']); ?></p>
<?php
                break;
        }
    }

    private function get_select_options($sub_field, $value)
    {
        $options = [];
        if ($sub_field['type'] === 'user') {
            $users = get_users(['fields' => ['ID', 'display_name']]);
            foreach ($users as $user) {
                $options[$user->ID] = $user->display_name;
            }
        } elseif ($sub_field['type'] === 'page_link' || $sub_field['type'] === 'post_object') {
            $args = [
                'post_type' => $sub_field['post_type'] ?? 'any',
                'posts_per_page' => -1,
                'post_status' => 'publish'
            ];
            $posts = get_posts($args);
            foreach ($posts as $post) {
                $options[$post->ID] = $post->post_title;
            }
        } elseif ($sub_field['type'] === 'relationship') {
            $args = [
                'post_type' => $sub_field['post_type'] ?? 'any',
                'posts_per_page' => -1,
                'post_status' => 'publish'
            ];
            $posts = get_posts($args);
            foreach ($posts as $post) {
                $options[$post->ID] = $post->post_title;
            }
        } elseif ($sub_field['type'] === 'taxonomy') {
            $terms = get_terms([
                'taxonomy' => $sub_field['taxonomy'] ?? 'category',
                'hide_empty' => false
            ]);
            if (!is_wp_error($terms)) {
                foreach ($terms as $term) {
                    $options[$term->term_id] = $term->name;
                }
            }
        }
        return $options;
    }

    public function ajax_get_options()
    {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';


        if (!wp_verify_nonce($nonce, 'acf_simple_repeater_nonce')) {

            wp_send_json_error(['message' => 'Nonce verification failed']);
            return;
        }

        $field_group_id = isset($_POST['field_group_id']) ? sanitize_text_field($_POST['field_group_id']) : '';
        $field_key = isset($_POST['field_key']) ? sanitize_text_field($_POST['field_key']) : '';
        $field_type = isset($_POST['field_type']) ? sanitize_text_field($_POST['field_type']) : '';
        $post_type = isset($_POST['post_type']) ? sanitize_text_field($_POST['post_type']) : '';
        $taxonomy = isset($_POST['taxonomy']) ? sanitize_text_field($_POST['taxonomy']) : '';



        if ($field_group_id) {
            $sub_fields = $this->get_sub_fields($field_group_id);
            if (empty($sub_fields)) {

                wp_send_json_error(['message' => 'No subfields found for group ID: ' . $field_group_id]);
                return;
            }
            wp_send_json_success(['subFields' => $sub_fields]);
            return;
        }

        $options = [];
        if ($field_type === 'user') {
            $users = get_users(['fields' => ['ID', 'display_name']]);
            foreach ($users as $user) {
                $options[$user->ID] = $user->display_name;
            }
        } elseif ($field_type === 'page_link' || $field_type === 'post_object') {
            $args = [
                'post_type' => $post_type ?: 'any',
                'posts_per_page' => -1,
                'post_status' => 'publish'
            ];
            $posts = get_posts($args);
            foreach ($posts as $post) {
                $options[$post->ID] = $post->post_title;
            }
        } elseif ($field_type === 'relationship') {
            $args = [
                'post_type' => $post_type ?: 'any',
                'posts_per_page' => -1,
                'post_status' => 'publish'
            ];
            $posts = get_posts($args);
            foreach ($posts as $post) {
                $options[$post->ID] = $post->post_title;
            }
        } elseif ($field_type === 'taxonomy') {
            $terms = get_terms([
                'taxonomy' => $taxonomy ?: 'category',
                'hide_empty' => false
            ]);
            if (!is_wp_error($terms)) {
                foreach ($terms as $term) {
                    $options[$term->term_id] = $term->name;
                }
            } else {
                error_log('ACF Simple Repeater: Taxonomy error: ' . $terms->get_error_message());
            }
        }

        if (empty($options) && $field_type) {

            wp_send_json_error(['message' => 'No options found for field type: ' . $field_type]);
            return;
        }

        wp_send_json_success(['options' => $options]);
    }

    public function input_admin_enqueue_scripts()
    {
        $url = plugin_dir_url(__FILE__);
        $version = '1.5.5';

        wp_enqueue_style('acf-simple-repeater', $url . 'assets/css/simple-repeater.css', [], $version);
        wp_enqueue_script('acf-simple-repeater', $url . 'assets/js/simple-repeater.js', ['jquery', 'acf-input', 'jquery-ui-datepicker', 'jquery-ui-timepicker'], $version, true);
        wp_enqueue_media();
        wp_enqueue_style('jquery-ui', '//code.jquery.com/ui/1.12.1/themes/base/jquery-ui.css', [], '1.12.1');
        wp_enqueue_script('jquery-ui-timepicker', '//cdnjs.cloudflare.com/ajax/libs/jquery-ui-timepicker-addon/1.6.3/jquery-ui-timepicker-addon.min.js', ['jquery-ui-datepicker'], '1.6.3', true);
        wp_enqueue_style('jquery-ui-timepicker', '//cdnjs.cloudflare.com/ajax/libs/jquery-ui-timepicker-addon/1.6.3/jquery-ui-timepicker-addon.min.css', [], '1.6.3');
        wp_localize_script('acf-simple-repeater', 'acfSimpleRepeater', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('acf_simple_repeater_nonce')
        ]);
    }
}

/**
 * Initialize plugin
 */
$acf_simple_repeater = new ACFSimpleRepeaterPlugin();
$acf_simple_repeater->init();

/**
 * Utility function to retrieve repeater values
 */
/*function get_simple_repeater_values($field_name, $post_id = null) {
    $value = get_field($field_name, $post_id);
    $data = $value ? json_decode($value, true) : [];
    if (!is_array($data)) {
        return [];
    }
    foreach ($data as &$row) {
        foreach ($row as $key => &$value) {
            $field_object = get_field_object($key);
            print_r($field_object);
            if ($field_object) {
                if ($field_object['type'] === 'image' || $field_object['type'] === 'file') {
                    $value = wp_get_attachment_url($value) ?: $value;
                } elseif ($field_object['type'] === 'user') {
                    $user = get_userdata($value);
                    $value = $user ? $user->display_name : $value;
                } elseif ($field_object['type'] === 'page_link' || $field_object['type'] === 'post_object') {
                    $post = get_post($value);
                    $value = $post ? $post->post_title : $value;
                } elseif ($field_object['type'] === 'relationship' || $field_object['type'] === 'taxonomy') {
                    if (is_array($value)) {
                        $formatted = [];
                        foreach ($value as $id) {
                            if ($field_object['type'] === 'taxonomy') {
                                $term = get_term($id);
                                $formatted[] = $term ? $term->name : $id;
                            } else {
                                $post = get_post($id);
                                $formatted[] = $post ? $post->post_title : $id;
                            }
                        }
                        $value = $formatted;
                    }
                }
            }
        }
    }
    return $data;
}*/
function get_simple_repeater_values($field_name, $post_id = null)
{
    $rows = get_field($field_name, $post_id);

    // If the repeater value is JSON (as in custom storage)
    if (is_string($rows)) {
        $rows = json_decode($rows, true);
    }

    if (!is_array($rows)) {
        return [];
    }

    $result = [];

    foreach ($rows as $row_index => $row) {
        $processed_row = [];

        foreach ($row as $field_key => $value) {
            $field_object = get_field_object($field_key, $post_id);

            if (!$field_object) continue;

            $field_name = $field_object['name'];
            $field_type = $field_object['type'];

            // === Convert ACF field value by type ===
            switch ($field_type) {
                case 'image':
                case 'file':
                    $value = ($value) ?: $value;
                    break;

                case 'user':
                    $user = get_userdata($value);
                    $value = $user ? $user->display_name : $value;
                    break;

                case 'page_link':
                case 'post_object':
                    $post = get_post($value);
                    $value = $post ? $post->post_title : $value;
                    break;

                case 'relationship':
                case 'taxonomy':
                    if (is_array($value)) {
                        $formatted = [];
                        foreach ($value as $id) {
                            if ($field_type === 'taxonomy') {
                                $term = get_term($id);
                                $formatted[] = $term ? $term->name : $id;
                            } else {
                                $post = get_post($id);
                                $formatted[] = $post ? $post->post_title : $id;
                            }
                        }
                        $value = $formatted;
                    }
                    break;
            }

            $processed_row[$field_name] = $value;
        }

        $result[] = $processed_row;
    }

    return $result;
}

function get_theme_option_repeater_values($field_name, $post_id = '')
{

    if (!empty(get_page_by_path('theme-settings')) && !$post_id) {
        $post_id = get_page_by_path('theme-settings')->ID;
    }
  
    if (!$post_id) {
        return array();
    }
    $rows = get_field($field_name, $post_id);

    // If the repeater value is JSON (as in custom storage)
    if (is_string($rows)) {
        $rows = json_decode($rows, true);
    }

    if (!is_array($rows)) {
        return [];
    }

    $result = [];

    foreach ($rows as $row_index => $row) {
        $processed_row = [];

        foreach ($row as $field_key => $value) {
            $field_object = get_field_object($field_key, $post_id);

            if (!$field_object) continue;

            $field_name = $field_object['name'];
            $field_type = $field_object['type'];

            // === Convert ACF field value by type ===
            switch ($field_type) {
                case 'image':
                case 'file':
                    $value = ($value) ?: $value;
                    break;

                case 'user':
                    $user = get_userdata($value);
                    $value = $user ? $user->display_name : $value;
                    break;

                case 'page_link':
                case 'post_object':
                    $post = get_post($value);
                    $value = $post ? $post->post_title : $value;
                    break;

                case 'relationship':
                case 'taxonomy':
                    if (is_array($value)) {
                        $formatted = [];
                        foreach ($value as $id) {
                            if ($field_type === 'taxonomy') {
                                $term = get_term($id);
                                $formatted[] = $term ? $term->name : $id;
                            } else {
                                $post = get_post($id);
                                $formatted[] = $post ? $post->post_title : $id;
                            }
                        }
                        $value = $formatted;
                    }
                    break;
            }

            $processed_row[$field_name] = $value;
        }

        $result[] = $processed_row;
    }

    return $result;
}
