<?php

/**
 * Luxurious Skincare Encoder IT functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Luxurious_Skincare_Encoder_IT
 */

if (! defined('_S_VERSION')) {
	// Replace the version number of the theme on each release.
	define('_S_VERSION', '1.0.0');
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function luxurious_skincare_encoder_it_setup()
{
	/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on Luxurious Skincare Encoder IT, use a find and replace
		* to change 'luxurious-skincare-encoder-it' to the name of your theme in all the template files.
		*/
	load_theme_textdomain('luxurious-skincare-encoder-it', get_template_directory() . '/languages');

	// Add default posts and comments RSS feed links to head.
	add_theme_support('automatic-feed-links');

	/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
	add_theme_support('title-tag');

	/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
	add_theme_support('post-thumbnails');

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'menu-1' => esc_html__('Primary', 'luxurious-skincare-encoder-it'),
		)
	);

	/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'luxurious_skincare_encoder_it_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support('customize-selective-refresh-widgets');

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);
}
add_action('after_setup_theme', 'luxurious_skincare_encoder_it_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function luxurious_skincare_encoder_it_content_width()
{
	$GLOBALS['content_width'] = apply_filters('luxurious_skincare_encoder_it_content_width', 640);
}
add_action('after_setup_theme', 'luxurious_skincare_encoder_it_content_width', 0);

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function luxurious_skincare_encoder_it_widgets_init()
{
	register_sidebar(
		array(
			'name'          => esc_html__('Sidebar', 'luxurious-skincare-encoder-it'),
			'id'            => 'sidebar-1',
			'description'   => esc_html__('Add widgets here.', 'luxurious-skincare-encoder-it'),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
}
add_action('widgets_init', 'luxurious_skincare_encoder_it_widgets_init');

/**
 * Enqueue scripts and styles.
 */
function luxurious_skincare_encoder_it_scripts()
{
	wp_enqueue_style('luxurious-skincare-encoder-it-style', get_stylesheet_uri(), array(), _S_VERSION);
	wp_style_add_data('luxurious-skincare-encoder-it-style', 'rtl', 'replace');

	wp_enqueue_script('luxurious-skincare-encoder-it-navigation', get_template_directory_uri() . '/js/navigation.js', array(), _S_VERSION, true);

	if (is_singular() && comments_open() && get_option('thread_comments')) {
		wp_enqueue_script('comment-reply');
	}
}
add_action('wp_enqueue_scripts', 'luxurious_skincare_encoder_it_scripts');

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if (defined('JETPACK__VERSION')) {
	require get_template_directory() . '/inc/jetpack.php';
}

/**
 * Load WooCommerce compatibility file.
 */
if (class_exists('WooCommerce')) {
	require get_template_directory() . '/inc/woocommerce.php';
}





/* Encoder Custom Code */
function luxurious_skincare_encoderit_scripts()
{
	wp_enqueue_style("luxurious-skincare-style", get_template_directory_uri() . "/assets/styles/luxurious-skincare-tailwind.css", false, false);
	wp_enqueue_style("custom-style", get_template_directory_uri() . "/assets/styles/luxurious-skincare-custom.css", false, false);


	wp_enqueue_script('custom_js', get_template_directory_uri() . '/assets/js/custom-script.js', array('jquery'), '', true);

	wp_enqueue_script('flowbite', get_template_directory_uri() . '/node_modules/flowbite/dist/flowbite.js', array(), '1.7.0', true);
}
add_action('wp_enqueue_scripts', 'luxurious_skincare_encoderit_scripts', 999);


//hide admin bar
function show_admin_bar_forcefully()
{
	return false;
}
add_filter('show_admin_bar', 'show_admin_bar_forcefully', 999999);


/**
 * HARD Protect Theme Settings Page
 * Prevent delete / trash completely
 */

define('THEME_SETTINGS_ID', 129);

/**
 * Block delete before it happens
 */
add_action('before_delete_post', function ($post_id) {
	if ($post_id == THEME_SETTINGS_ID) {
		wp_die(
			__('This page is protected and cannot be deleted.', 'luxurious-skincare-encoder-it'),
			__('Action blocked', 'luxurious-skincare-encoder-it'),
			['response' => 403]
		);
	}
});

/**
 * Block moving to trash
 */
add_filter('pre_trash_post', function ($trash, $post) {
	if ($post->ID == THEME_SETTINGS_ID) {
		wp_die(
			__('This page is protected and cannot be moved to trash.', 'luxurious-skincare-encoder-it'),
			__('Action blocked', 'luxurious-skincare-encoder-it'),
			['response' => 403]
		);
	}
	return $trash;
}, 10, 2);

/**
 * Remove Trash / Delete links from Pages list (UI safety)
 */
add_filter('post_row_actions', function ($actions, $post) {
	if ($post->ID == THEME_SETTINGS_ID) {
		unset($actions['trash']);
		unset($actions['delete']);
	}
	return $actions;
}, 10, 2);

/**
 * Lock page title (prevent rename)
 */
add_action('admin_head', function () {
	if (
		isset($_GET['post']) &&
		$_GET['post'] == THEME_SETTINGS_ID
	) {
		echo '<style>
            #title {
                pointer-events: none;
                background: #f5f5f5;
            }
        </style>';
	}
});

add_action('admin_menu', function () {

	add_menu_page(
		'Theme Settings',                 // Page title
		'Theme Settings',                 // Menu title
		'edit_pages',                     // ✅ CORRECT capability
		'post.php?post=129&action=edit',  // Direct edit URL
		'',                               // No callback
		'dashicons-admin-generic',        // Icon
		20                                // Position
	);
});


/**
 * Clean Theme Settings page UI
 * Remove default WordPress features (ACF Free)
 */

add_action('admin_init', function () {

	if (
		isset($_GET['post']) &&
		(int) $_GET['post'] === THEME_SETTINGS_ID
	) {

		remove_post_type_support('page', 'editor');
		remove_post_type_support('page', 'title');
		remove_post_type_support('page', 'thumbnail');
		remove_post_type_support('page', 'page-attributes');
		remove_post_type_support('page', 'comments');
		remove_post_type_support('page', 'custom-fields');
	}
});

function get_youtube_poster($url)
{
	preg_match(
		'/(?:youtube\.com\/(?:.*v=|embed\/)|youtu\.be\/)([^&?\/]+)/',
		$url,
		$matches
	);

	if (!isset($matches[1])) return false;

	$id = $matches[1];

	return [
		'max' => "https://img.youtube.com/vi/$id/maxresdefault.jpg",
		'hq'  => "https://img.youtube.com/vi/$id/hqdefault.jpg",
	];
}
function acf_oembed_get_url($iframe)
{
	if (!$iframe) return false;

	preg_match('/src="(.+?)"/', $iframe, $matches);

	return $matches[1] ?? false;
}
function get_product_reviews_formatted($product_id, $limit = 6)
{
	if (!$product_id) return [];

	$args = [
		'post_id' => $product_id,
		'status'  => 'approve',
		'number'  => $limit,
		'type'    => 'review',
	];

	$comments = get_comments($args);
	$reviews  = [];

	foreach ($comments as $comment) {
		$is_verified = get_comment_meta($comment->comment_ID, 'verified', true);

		$reviews[] = [
			'name'   => $comment->comment_author,
			'role'   => $is_verified ? 'Verified Purchase' : 'Customer',
			'review' => wp_trim_words(
				wp_strip_all_tags($comment->comment_content),
				40,
				'...'
			),
		];
	}

	return $reviews;
}
function get_custom_reviews_prepared($limit = 6)
{
	$args = [
		'post_type'      => 'reviews',
		'posts_per_page' => $limit,
		'post_status'    => 'publish',
	];

	$query = new WP_Query($args);
	$reviews = [];

	if ($query->have_posts()) {
		while ($query->have_posts()) {
			$query->the_post();
			$post_id = get_the_ID();
			$reviews = get_field('reviews');
			$imageid = $reviews['reviewer_image'];
			$imageurl = wp_get_attachment_image_url($imageid, 'full');
			$reviewsdata[] = [
				'post_id'   => $post_id,
				'name'   => $reviews['reviewer_name'],
				'date'  => get_the_date('d M Y', $post_id),
				'image' => ($imageurl) ? $imageurl : get_template_directory_uri() . '/assets/images/default-users.png',
				'role'   => $reviews['reviewer_role'] ?: 'Customer',
				'review' => $reviews['review_text'],
				'text' => $reviews['review_text'],
				'rating' => $reviews['rating'],
			];
		}
		wp_reset_postdata();
	}

	return $reviewsdata;
}
function get_primary_menu_items()
{
	$locations = get_nav_menu_locations();

	if (!isset($locations['menu-1'])) return [];

	$menu = wp_get_nav_menu_object($locations['menu-1']);
	if (!$menu) return [];

	$items = wp_get_nav_menu_items($menu->term_id);
	if (!$items) return [];

	return $items;
}

//reviews
function register_reviews_cpt()
{
	register_post_type('reviews', [
		'labels' => [
			'name'          => 'Reviews',
			'singular_name' => 'Review',
			'add_new_item'  => 'Add Review',
			'edit_item'     => 'Edit Review',
		],
		'public'        => false,
		'show_ui'       => true,
		'menu_icon'     => 'dashicons-testimonial',
		'supports'      => ['title'],
		'capability_type' => 'post',
	]);
}
add_action('init', 'register_reviews_cpt');

//blog ajax
add_action('wp_ajax_load_more_blogs', 'load_more_blogs');
add_action('wp_ajax_nopriv_load_more_blogs', 'load_more_blogs');

function load_more_blogs()
{

	$page = isset($_POST['page']) ? intval($_POST['page']) : 1;

	$featured_posts = (array) get_field('blog_page_top_post', THEME_SETTINGS_ID);
	$featured_posts = array_filter(array_map('intval', $featured_posts));

	$posts_per_page = 6;

	$args = [
		'post_type'      => 'post',
		'posts_per_page' => $posts_per_page,
		'post__not_in'   => $featured_posts,
		'orderby'        => 'date',
		'order'          => 'DESC',
		'offset'         => $page * $posts_per_page,
	];

	$query = new WP_Query($args);

	if ($query->have_posts()) :
		while ($query->have_posts()) : $query->the_post();
			get_template_part(
				'template-parts/blog/blog-card',
				null,
				['blog_id' => get_the_ID()]
			);
		endwhile;
		wp_reset_postdata();
	endif;

	wp_die();
}
//cart fargment
add_filter('woocommerce_add_to_cart_fragments', function ($fragments) {

	ob_start();
?>
	<span class="bg-(--primary-marigold) rounded-full h-4 min-w-4 flex items-center justify-center absolute -top-0.5 -right-0.5 text-(--phthalo-green) text-center text-[10px] font-semibold py-0.5 px-1">
		<?php echo WC()->cart->get_cart_contents_count(); ?>
	</span>
<?php
	$fragments['.cart-target span'] = ob_get_clean();

	return $fragments;
});

//add to cart
add_action('wp_enqueue_scripts', function () {
	wp_enqueue_script( 'wc-cart-fragments' );
	if (!class_exists('WooCommerce')) return;

	wp_enqueue_script(
		'theme-woo-ajax',
		get_template_directory_uri() . '/assets/js/woo-ajax.js',
		['jquery'],
		'1.0',
		true
	);

	$data = [
		'ajax_url'    => admin_url('admin-ajax.php'),
		'wc_ajax_url' => WC_AJAX::get_endpoint('%%endpoint%%'),
		'nonce'       => wp_create_nonce('woo_ajax_nonce'),
	];
	$productid = get_field('shop_product', THEME_SETTINGS_ID);
	$data['checkout_url'] = wc_get_checkout_url();
	// Pass product_id ONLY on single product page
	if (is_product()) {
		global $post;
		if ($post) {
			$data['product_id'] = $post->ID;
		}
	} else {
		$productid = get_field('shop_product', THEME_SETTINGS_ID);
		$data['product_id'] = $productid;
	}


	wp_localize_script('theme-woo-ajax', 'wooAjax', $data);
});

add_action('wp_ajax_woocommerce_custom_add_to_cart', 'woocommerce_custom_add_to_cart');
add_action('wp_ajax_nopriv_woocommerce_custom_add_to_cart', 'woocommerce_custom_add_to_cart');

function woocommerce_custom_add_to_cart()
{

	if (! isset($_POST['product_id'], $_POST['variation_id'], $_POST['quantity'])) {
		wp_send_json_error(['message' => 'Missing parameters']);
	}

	$product_id   = absint($_POST['product_id']);
	$variation_id = absint($_POST['variation_id']);
	$quantity     = max(1, absint($_POST['quantity']));

	$product   = wc_get_product($product_id);
	$variation = wc_get_product($variation_id);

	if (! $product || ! $variation || ! $variation->is_type('variation')) {
		wp_send_json_error(['message' => 'Invalid product or variation']);
	}

	// 🔒 Get variation attributes directly from the variation itself
	$variation_attributes = $variation->get_attributes();

	// 🔒 Stock check
	if (! $variation->is_in_stock()) {
		wp_send_json_error(['message' => 'Variation out of stock']);
	}

	// 🛒 Add to cart (bypasses Woo form validation)
	$cart_item_key = WC()->cart->add_to_cart(
		$product_id,
		$quantity,
		$variation_id,
		$variation_attributes
	);

	if (! $cart_item_key) {
		wp_send_json_error(['message' => 'Failed to add to cart']);
	}


	ob_start();
	WC_AJAX::get_refreshed_fragments();
	$fragments_json = ob_get_clean();

	$fragments_data = json_decode($fragments_json, true);

	wp_send_json_success([
		'message'   => 'Added to cart',
		'cart_key'  => $cart_item_key,
		'fragments' => $fragments_data['fragments'],
		'cart_hash' => $fragments_data['cart_hash'],
	]);
	// 🔄 Get updated fragments

}

//reviews
function reviews_data()
{
	$reviews = get_field('reviews_data', THEME_SETTINGS_ID);
	return $reviews;
}
